package com.brandon3055.draconicevolution.blocks.tileentity;

import codechicken.lib.raytracer.RayTracer;
import codechicken.lib.raytracer.SubHitBlockHitResult;
import com.brandon3055.brandonscore.api.power.IOPStorage;
import com.brandon3055.brandonscore.blocks.TileBCore;
import com.brandon3055.brandonscore.capability.CapabilityOP;
import com.brandon3055.brandonscore.inventory.TileItemStackHandler;
import com.brandon3055.brandonscore.lib.IInteractTile;
import com.brandon3055.brandonscore.lib.IRSSwitchable;
import com.brandon3055.brandonscore.lib.datamanager.DataFlags;
import com.brandon3055.brandonscore.lib.datamanager.ManagedBool;
import com.brandon3055.brandonscore.lib.datamanager.ManagedEnum;
import com.brandon3055.brandonscore.utils.EnergyUtils;
import com.brandon3055.draconicevolution.init.DEContent;
import com.brandon3055.draconicevolution.inventory.TransfuserMenu;
import com.brandon3055.draconicevolution.utils.ItemCapMerger;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.InteractionHand;
import net.minecraft.world.InteractionResult;
import net.minecraft.world.ItemInteractionResult;
import net.minecraft.world.MenuProvider;
import net.minecraft.world.entity.player.Inventory;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.inventory.AbstractContainerMenu;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.phys.BlockHitResult;
import net.minecraft.world.phys.HitResult;
import net.neoforged.neoforge.capabilities.Capabilities;
import net.neoforged.neoforge.capabilities.RegisterCapabilitiesEvent;
import net.neoforged.neoforge.items.IItemHandler;
import net.neoforged.neoforge.items.IItemHandlerModifiable;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * Created by brandon3055 on 12/12/2020.
 */
public class TileEnergyTransfuser extends TileBCore implements IInteractTile, MenuProvider, IRSSwitchable {

    public TileItemStackHandler itemNorth = new TileItemStackHandler(this, 1).setSlotLimit(1).setStackValidator(EnergyUtils::isEnergyItem);
    public TileItemStackHandler itemEast = new TileItemStackHandler(this, 1).setSlotLimit(1).setStackValidator(EnergyUtils::isEnergyItem);
    public TileItemStackHandler itemSouth = new TileItemStackHandler(this, 1).setSlotLimit(1).setStackValidator(EnergyUtils::isEnergyItem);
    public TileItemStackHandler itemWest = new TileItemStackHandler(this, 1).setSlotLimit(1).setStackValidator(EnergyUtils::isEnergyItem);
    private IItemHandlerModifiable[] indexedItemHandlers = {itemNorth, itemEast, itemSouth, itemWest};
    public IItemHandlerModifiable itemsCombined = (IItemHandlerModifiable) ItemCapMerger.merge(itemNorth, itemEast, itemSouth, itemWest);
    public IOPStorage opStorage = new OPIOAdapter();
    @SuppressWarnings("unchecked")
    public ManagedEnum<ItemIOMode>[] ioModes = new ManagedEnum[4]; //North, East, South, West
    public ManagedBool balancedMode = register(new ManagedBool("balance_mode", DataFlags.SAVE_NBT_SYNC_CONTAINER, DataFlags.CLIENT_CONTROL));

    public TileEnergyTransfuser(BlockPos pos, BlockState state) {
        super(DEContent.TILE_ENERGY_TRANSFUSER.get(), pos, state);
        capManager.setInternalManaged("item_north", Capabilities.ItemHandler.BLOCK, itemNorth).syncTile().saveBoth();
        capManager.setInternalManaged("item_east", Capabilities.ItemHandler.BLOCK, itemEast).syncTile().saveBoth();
        capManager.setInternalManaged("item_south", Capabilities.ItemHandler.BLOCK, itemSouth).syncTile().saveBoth();
        capManager.setInternalManaged("item_west", Capabilities.ItemHandler.BLOCK, itemWest).syncTile().saveBoth();
        capManager.set(CapabilityOP.BLOCK, opStorage, Direction.UP, Direction.DOWN, null);

        capManager.set(Capabilities.ItemHandler.BLOCK, new ItemIOAdapter(0, itemNorth), Direction.NORTH);
        capManager.set(Capabilities.ItemHandler.BLOCK, new ItemIOAdapter(1, itemEast), Direction.EAST);
        capManager.set(Capabilities.ItemHandler.BLOCK, new ItemIOAdapter(2, itemSouth), Direction.SOUTH);
        capManager.set(Capabilities.ItemHandler.BLOCK, new ItemIOAdapter(3, itemWest), Direction.WEST);
        capManager.set(Capabilities.ItemHandler.BLOCK, new ItemIOAdapter(-1, itemsCombined), Direction.UP, Direction.DOWN, null);

        for (int i = 0; i < 4; i++) {
            ioModes[i] = register(new ManagedEnum<>("item_mode_" + i, ItemIOMode.CHARGE, DataFlags.SAVE_BOTH_SYNC_TILE, DataFlags.CLIENT_CONTROL));
        }
    }

    public static void register(RegisterCapabilitiesEvent event) {
        energyCapability(event, DEContent.TILE_ENERGY_TRANSFUSER);
        capability(event, DEContent.TILE_ENERGY_TRANSFUSER, Capabilities.ItemHandler.BLOCK);
    }

    @Override
    public void tick() {
        super.tick();
        if (level.isClientSide || !isTileEnabled()) {
            return;
        }

        for (int i = 0; i < 4; i++) {
            ItemIOMode sourcemode = ioModes[i].get();
            if (sourcemode.discharge) {
                ItemStack stack = itemsCombined.getStackInSlot(i);
                IOPStorage sourceStorage = EnergyUtils.getStorage(stack);
                if (sourceStorage == null) continue;

                boolean canExtract = sourceStorage.canExtract();
                //I want to be able to discharge DE tools, armor, etc but i dont want them to be usable as buffer items.
                boolean extractOverride = !canExtract && sourcemode == ItemIOMode.DISCHARGE;
                if (canExtract || extractOverride) {
                    long maxExtract;
                    if (extractOverride) {
                        maxExtract = Math.min(sourceStorage.maxExtract(), sourceStorage.getOPStored());
                    } else {
                        maxExtract = sourceStorage.extractOP(sourceStorage.getOPStored(), true);
                    }
                    if (maxExtract == 0) continue;

                    long totalSent = 0;
                    long sent = sendEnergyTo(maxExtract, Direction.DOWN);
                    maxExtract -= sent;
                    totalSent += sent;
                    sent = sendEnergyTo(maxExtract, Direction.UP);
                    maxExtract -= sent;
                    totalSent += sent;

                    if (maxExtract > 0) {
                        for (int j = 0; j < 4; j++) {
                            if (j == i || maxExtract == 0) continue;
                            ItemIOMode targetMode = ioModes[j].get();
                            //We dont want to bother sending power between buffers.
                            if (targetMode.charge && (targetMode == ItemIOMode.CHARGE || sourcemode == ItemIOMode.DISCHARGE)) {
                                sent = EnergyUtils.insertEnergy(itemsCombined.getStackInSlot(j), maxExtract, false);
                                maxExtract -= sent;
                                totalSent += sent;
                            }
                        }
                    }

                    if (extractOverride) {
                        sourceStorage.modifyEnergyStored(-totalSent);
                    } else {
                        sourceStorage.extractOP(totalSent, false);
                    }
                }
            }
        }
    }

    @Override
    public ItemInteractionResult useItemOn(ItemStack heldStack, BlockState state, Player player, InteractionHand hand, BlockHitResult hitIn) {
        if (level.isClientSide) {
            return ItemInteractionResult.SUCCESS;
        }

        HitResult hit = RayTracer.retrace(player);
        int slot = hit instanceof SubHitBlockHitResult ? ((SubHitBlockHitResult) hit).subHit : -1;
        if (slot > -1 && slot < 4 && hand != null) {
            ItemStack stack = itemsCombined.getStackInSlot(slot);
            if (!stack.isEmpty() && heldStack.isEmpty()) {
                player.setItemInHand(hand, stack);
                itemsCombined.setStackInSlot(slot, ItemStack.EMPTY);
                return ItemInteractionResult.SUCCESS;
            } else if (stack.isEmpty() && !heldStack.isEmpty()) {
                if (itemsCombined.isItemValid(slot, heldStack)) {
                    if (heldStack.getCount() > 1) {
                        ItemStack copy = heldStack.copy();
                        copy.setCount(1);
                        itemsCombined.setStackInSlot(slot, copy);
                        heldStack.shrink(1);
                    } else {
                        itemsCombined.setStackInSlot(slot, heldStack);
                        player.setItemInHand(hand, ItemStack.EMPTY);
                    }
                    return ItemInteractionResult.SUCCESS;
                }
            }
        }

        if (player instanceof ServerPlayer) {
            player.openMenu(this, worldPosition);
        }
        return ItemInteractionResult.SUCCESS;
    }

    @Nullable
    @Override
    public AbstractContainerMenu createMenu(int id, Inventory inv, Player player) {
        return new TransfuserMenu(id, player.getInventory(), this);
    }

    public enum ItemIOMode {
        /**
         * - Charge from External
         * - Charge from Buffer
         * - Can be extracted when full
         */
        CHARGE(0, "mode_charge", true, false, 0xFF8500),
        /**
         * - Discharge to External
         * - Discharge to Buffer
         * - Can be extracted when empty
         */
        DISCHARGE(1, "mode_discharge", false, true, 0x0050FF),
        /**
         * - Charge from External
         * - Charge from Discharge Slots
         * - Discharge to External
         * - Discharge to Charge Slots
         * - Can not be extracted
         */
        BUFFER(2, "mode_buffer", true, true, 0xFF00FF),
        DISABLED(3, "mode_disabled", false, false, 0x202020); //Also disables slots

        private int index;
        private String name;
        public final boolean discharge;
        public final boolean charge;
        private int colour;

        ItemIOMode(int index, String name, boolean charge, boolean discharge, int colour) {
            this.index = index;
            this.name = name;
            this.discharge = discharge;
            this.charge = charge;
            this.colour = colour;
        }

        public ItemIOMode nextMode(boolean previous) {
            return values()[Math.floorMod(index + (previous ? -1 : 1), values().length)];
        }

        public String getName() {
            return name;
        }

        public String getSpriteName() {
            return "transfuser/" + name;
        }

        public boolean canExtract(IOPStorage storage) {
            return switch (this) {
                case CHARGE -> storage.getOPStored() >= storage.getMaxOPStored();
                case DISCHARGE -> storage.getOPStored() == 0;
                default -> false;
            };
        }

        public int getColour() {
            return colour;
        }
    }

    public class OPIOAdapter implements IOPStorage {

        @Override
        public long receiveOP(long maxReceive, boolean simulate) {
            if (!isTileEnabled()) return 0;
            long totalAccepted = 0;
            for (int i = 0; i < 4; i++) {
                ItemIOMode mode = ioModes[i].get();
                if (mode.charge) {
                    ItemStack stack = itemsCombined.getStackInSlot(i);
                    long accepted;
                    if (balancedMode.get()) {
                        accepted = EnergyUtils.insertEnergy(stack, maxReceive / 4, simulate);
                    } else {
                        accepted = EnergyUtils.insertEnergy(stack, maxReceive, simulate);
                        maxReceive -= accepted;
                    }
                    totalAccepted += accepted;
                }
            }
            return totalAccepted;
        }

        @Override
        public long extractOP(long maxExtract, boolean simulate) {
            if (!isTileEnabled()) return 0;
            long totalExtracted = 0;
            for (int i = 0; i < 4; i++) {
                ItemIOMode mode = ioModes[i].get();
                if (mode.discharge) {
                    ItemStack stack = itemsCombined.getStackInSlot(i);
                    long extracted = EnergyUtils.extractEnergy(stack, maxExtract, simulate);
                    totalExtracted += extracted;
                    maxExtract -= extracted;
                }
            }
            return totalExtracted;
        }

        @Override
        public long getOPStored() {
            long total = 0;
            for (int i = 0; i < 4; i++) {
                total += EnergyUtils.getEnergyStored(itemsCombined.getStackInSlot(i));
            }
            return total;
        }

        @Override
        public long getMaxOPStored() {
            long total = 0;
            for (int i = 0; i < 4; i++) {
                total += EnergyUtils.getMaxEnergyStored(itemsCombined.getStackInSlot(i));
            }
            return total;
        }

        @Override
        public boolean canExtract() {
            if (!isTileEnabled()) return false;
            for (ManagedEnum<ItemIOMode> mode : ioModes) {
                if (mode.get().discharge) return true;
            }
            return false;
        }

        @Override
        public boolean canReceive() {
            if (!isTileEnabled()) return false;
            for (ManagedEnum<ItemIOMode> mode : ioModes) {
                if (mode.get().charge) return true;
            }
            return false;
        }

        @Override
        public long modifyEnergyStored(long amount) {
            return 0; //Invalid operation for this device
        }
    }

    public class ItemIOAdapter implements IItemHandler {
        private int index;
        private IItemHandler handler;

        public ItemIOAdapter(int index, IItemHandler handler) {
            this.index = index;
            this.handler = handler;
        }

        @Override
        public int getSlots() {
            return handler.getSlots();
        }

        @NotNull
        @Override
        public ItemStack getStackInSlot(int slot) {
            return handler.getStackInSlot(slot);
        }

        @NotNull
        @Override
        public ItemStack insertItem(int slot, @NotNull ItemStack stack, boolean simulate) {
            ItemIOMode mode = getMode(slot);
            if (mode == ItemIOMode.DISABLED) {
                return stack;
            }
            return handler.insertItem(slot, stack, simulate);
        }

        @NotNull
        @Override
        public ItemStack extractItem(int slot, int amount, boolean simulate) {
            ItemIOMode mode = getMode(slot);
            IOPStorage storage = EnergyUtils.getStorage(handler.getStackInSlot(slot));
            if (storage != null && mode.canExtract(storage)) {
                return handler.extractItem(slot, amount, simulate);
            }
            return ItemStack.EMPTY;
        }

        @Override
        public int getSlotLimit(int slot) {
            return handler.getSlotLimit(slot);
        }

        @Override
        public boolean isItemValid(int slot, @NotNull ItemStack stack) {
            return handler.isItemValid(slot, stack);
        }

        private ItemIOMode getMode(int slot) {
            int i = index != -1 ? index : slot;
            return ioModes[i].get();
        }
    }

}
