package codechicken.multipart.util;

import codechicken.lib.data.MCDataByteBuf;
import codechicken.multipart.api.TickableTile;
import codechicken.multipart.block.TileMultipart;
import codechicken.multipart.init.CBMultipartModContent;
import codechicken.multipart.network.MultiPartSPH;
import io.netty.buffer.Unpooled;
import net.covers1624.quack.util.CrashLock;
import net.minecraft.core.BlockPos;
import net.minecraft.core.HolderLookup;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.world.level.block.entity.BlockEntity;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.chunk.LevelChunk;
import net.neoforged.bus.api.EventPriority;
import net.neoforged.neoforge.common.NeoForge;
import net.neoforged.neoforge.event.level.ChunkEvent;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * Created by covers1624 on 13/5/20.
 */
public class MultipartLoadHandler {

    private static final Logger logger = LogManager.getLogger();
    private static final CrashLock LOCK = new CrashLock("Already initialized.");

    public static void init() {
        LOCK.lock();

        NeoForge.EVENT_BUS.addListener(EventPriority.HIGHEST, MultipartLoadHandler::onChunkLoad);
    }

    // TODO move this to a Mixin.
    // Vanilla fires BlockEntity.handleUpdateTag before the LevelChunk has been added to the world.
    private static void onChunkLoad(ChunkEvent.Load event) {
        if (event.getLevel().isClientSide() && event.getChunk() instanceof LevelChunk chunk) {
            for (BlockEntity be : List.copyOf(chunk.getBlockEntities().values())) {
                if (be instanceof TileNBTContainer tile && tile.updateTag != null) {
                    byte[] data = tile.updateTag.getByteArray("data");
                    TileMultipart.handleDescPacket(tile.getLevel(), tile.getBlockPos(), new MCDataByteBuf(Unpooled.wrappedBuffer(data)));
                }
            }
        }
    }

    //This is a fallback in the event that our Mixin does not get hit.
    public static class TileNBTContainer extends BlockEntity implements TickableTile {

        //Store the number of ticks this tile has existed for.
        //We use this to remove the tile from the ticking list
        //after it has existed for too long.
        private int ticks;
        //If the tile has taken too long to load.
        private boolean failed;
        //If the tile has successfully loaded.
        //Here just in case something weird happens,
        //we don't load it multiple times.
        private boolean loaded;
        //The NBT of the tile.
        //We save this back out in case something breaks.
        @Nullable
        public CompoundTag tag;

        @Nullable
        public CompoundTag updateTag;

        public TileNBTContainer(BlockPos pos, BlockState state) {
            super(CBMultipartModContent.MULTIPART_TILE_TYPE.get(), pos, state);
        }

        //Handle initial desc sync
        @Override
        public void handleUpdateTag(CompoundTag tag, HolderLookup.Provider registries) {
            if (!tag.contains("data")) {
                logger.warn("Received update tag without 'data' field. Ignoring..");
                return;
            }
            updateTag = tag;
        }

        @Override
        public void loadAdditional(CompoundTag compound, HolderLookup.Provider registries) {
            super.loadAdditional(compound, registries);
            tag = compound.copy();
        }

        @Override
        public void saveAdditional(CompoundTag compound, HolderLookup.Provider registries) {
            super.saveAdditional(compound, registries);
            if (tag != null) {
                compound.merge(tag);
            }
        }

        @Override
        public void tick() {
            if (level == null || level.isClientSide) {
                return;
            }

            if (!failed && !loaded) {
                if (tag != null) {
                    TileMultipart newTile = TileMultipart.fromNBT(tag, getBlockPos(), getLevel().registryAccess());
                    if (newTile != null) {
                        newTile.clearRemoved();
                        level.setBlockEntity(newTile);
                        newTile.notifyTileChange();
                        newTile.notifyShapeChange();
                        MultiPartSPH.sendDescUpdate(newTile);
                    } else {
                        level.removeBlock(getBlockPos(), false);
                    }
                    loaded = true;
                } else {
                    ticks += 1;
                    if ((ticks % 600) == 0) {
                        failed = true;
                        logger.warn("TileNBTContainer at '{}' still exists after {} ticks! Deleting..", getBlockPos(), ticks);
                        level.removeBlock(getBlockPos(), false);
                    }
                }
            }
        }
    }
}
