package net.covers1624.coffeegrinder.bytecode.matching;

import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.InstructionFlag;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import org.jetbrains.annotations.Nullable;

import static java.util.Objects.requireNonNull;
import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.matchLoadLocal;

/**
 * Created by covers1624 on 19/4/21.
 */
public class BranchLeaveMatching {

    @Nullable
    public static Return matchReturn(@Nullable Instruction insn, MethodDecl method) {
        if (!(insn instanceof Return ret)) return null;

        return ret.getMethod() == method ? ret : null;
    }

    /**
     * Attempts to match a leave instruction which leaves the specified BlockContainer.
     *
     * @param insn      The instruction.
     * @param container The BlockContainer.
     * @return The Leave instruction, otherwise empty.
     */
    @Nullable
    public static Leave matchLeave(@Nullable Instruction insn, BlockContainer container) {
        if (!(insn instanceof Leave leave)) return null;

        if (leave.getTargetContainer() != container) return null;
        return leave;
    }

    /**
     * Attempts to match a branch instruction to a given block.
     *
     * @param insn        The instruction.
     * @param targetBlock The target block.
     * @return The Branch instruction, otherwise empty.
     */
    @Nullable
    public static Branch matchBranch(@Nullable Instruction insn, Block targetBlock) {
        if (!(insn instanceof Branch branch)) return null;
        if (branch.getTargetBlock() != targetBlock) return null;
        return branch;
    }

    /**
     * Returns true if exit1 and exit2 are both exit instructions
     * (branch or leave) and both represent the same exit.
     *
     * @param exit1 The first exit point.
     * @param exit2 The second exit point.
     * @return If they represent the same exit.
     */
    public static boolean compatibleExitInstruction(@Nullable Instruction exit1, @Nullable Instruction exit2) {
        return switch (exit1) {
            case Branch br1 when exit2 instanceof Branch br2 -> br1.getTargetBlock() == br2.getTargetBlock();
            case Leave l1 when exit2 instanceof Leave l2 -> l1.getTargetContainer() == l2.getTargetContainer();
            case null, default -> false;
        };
    }

    /**
     * Attempts to match a {@link Throw} instruction, which loads
     * the specified variable.
     *
     * @param insn     The Instruction.
     * @param variable The variable.
     * @return The {@link Throw} or <code>null</code>.
     */
    @Nullable
    public static Throw matchThrow(@Nullable Instruction insn, LocalVariable variable) {
        if (!(insn instanceof Throw thr)) return null;

        if (matchLoadLocal(thr.getArgument(), variable) == null) return null;

        return thr;
    }

    /**
     * Determine if the specified instruction necessarily exits (END_POINT_UNREACHABLE)
     * and if so, return last (or single) exit instruction.
     *
     * @param insn The instruction to check.
     * @return The exit instruction, or <code>null</code>.
     */
    @Nullable
    public static Instruction tryGetExit(@Nullable Instruction insn) {
        Instruction exitInsn = insn;
        if (insn instanceof Block) {
            exitInsn = insn.getLastChildOrNull();
        }
        if (exitInsn != null && exitInsn.hasFlag(InstructionFlag.END_POINT_UNREACHABLE)) {
            return exitInsn;
        }
        return null;
    }

    /**
     * Gets the final instruction from a block (or a single instruction) assuming that all blocks
     * or instructions in this position have unreachable endpoints.
     *
     * @param insn The insn or block to check.
     * @return The exit insn.
     */
    public static Instruction getExit(Instruction insn) {
        return requireNonNull(tryGetExit(insn));
    }
}
