package net.covers1624.coffeegrinder.bytecode.transform.transformers;

import net.covers1624.coffeegrinder.bytecode.SimpleInsnVisitor;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.transform.MethodTransformContext;
import net.covers1624.coffeegrinder.bytecode.transform.MethodTransformer;
import net.covers1624.coffeegrinder.util.None;

import static java.util.Objects.requireNonNull;
import static net.covers1624.coffeegrinder.bytecode.matching.BranchLeaveMatching.matchLeave;
import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.matchLoadLocal;

/**
 * Transforms MonitorEnter/Exit + Try-Finally into a Synchronized block.
 * <p>
 * Created by covers1624 on 8/7/21.
 */
public class SynchronizedTransform extends SimpleInsnVisitor<MethodTransformContext> implements MethodTransformer {

    @Override
    public void transform(MethodDecl function, MethodTransformContext ctx) {
        function.accept(this, ctx);
    }

    @Override
    public None visitTryCatch(TryCatch tryCatch, MethodTransformContext ctx) {
        if (tryCatch.getFinallyBody() == null || !(tryCatch.getPrevSiblingOrNull() instanceof MonitorEnter monitor)) return super.visitTryCatch(tryCatch, ctx);

        // MONITOR_ENTER(STORE finallyVar, ...)
        // TRY_FINALLY {
        //   ...
        // } finally {
        //   MONITOR_EXIT(LOAD finallyVar)
        //   LEAVE finally
        // }
        // ->
        // SYNCHRONIZED (...) {
        //   ...
        // }
        ctx.pushStep("Generate Synchronized");

        Store store = (Store) monitor.getArgument();

        Block finallyBody = tryCatch.getFinallyBody().getEntryPoint();
        MonitorExit monitorExit = (MonitorExit) finallyBody.getFirstChild();

        requireNonNull(matchLoadLocal(monitorExit.getArgument(), store.getVariable()));
        requireNonNull(matchLeave(monitorExit.getNextSibling(), tryCatch.getFinallyBody()));

        Synchronized sync = tryCatch.replaceWith(new Synchronized(store.getValue(), tryCatch.getTryBody()).withOffsets(tryCatch));

        assert store.getVariable().getLoadCount() == 0;
        monitor.remove();
        ctx.popStep();

        sync.accept(this, ctx);
        return NONE;
    }
}

