package net.covers1624.coffeegrinder.bytecode.insns;

import net.covers1624.coffeegrinder.bytecode.*;
import net.covers1624.coffeegrinder.type.AType;
import net.covers1624.coffeegrinder.type.PrimitiveType;
import net.covers1624.coffeegrinder.util.EnumBitSet;
import org.jetbrains.annotations.Nullable;

/**
 * Created by covers1624 on 5/5/21.
 */
public final class SwitchTable extends Instruction {

    private final InstructionSlot<Instruction> value = new InstructionSlot<>(this);
    public final InstructionCollection<SwitchSection> sections = new InstructionCollection<>(this);
    public final @Nullable TableInfo tableInfo;

    public SwitchTable(Instruction value, @Nullable TableInfo tableInfo) {
        this.value.set(value);
        this.tableInfo = tableInfo;
    }

    @Override
    public AType getResultType() {
        return PrimitiveType.VOID;
    }

    @Override
    public EnumBitSet<InstructionFlag> getDirectFlags() {
        return InstructionFlag.NONE;
    }

    @Override
    protected EnumBitSet<InstructionFlag> computeFlags() {
        EnumBitSet<InstructionFlag> flags = getDirectFlags().copy();

        for (SwitchSection section : sections) {
            flags.or(section.getFlags());
        }

        flags.or(value.get().getFlags());
        return flags;
    }

    @Override
    public <R, C> R accept(InsnVisitor<R, C> visitor, C ctx) {
        return visitor.visitSwitchTable(this, ctx);
    }

    //@formatter:off
    public Instruction getValue() { return value.get(); }
    public void setValue(Instruction value) { this.value.set(value); }
    //@formatter:on

    public boolean isExhaustive() {
        return sections.anyMatch(section -> section.values.anyMatch(e -> e instanceof Nop));
    }

    public static final class SwitchSection extends Instruction {

        public final InstructionCollection<Instruction> values = new InstructionCollection<>(this);

        private final InstructionSlot<Instruction> body = new InstructionSlot<>(this);

        public boolean explicitBlock;

        public SwitchSection(Instruction body) {
            this.body.set(body);
        }

        @Override
        public AType getResultType() {
            return PrimitiveType.VOID;
        }

        @Override
        public EnumBitSet<InstructionFlag> getDirectFlags() {
            return InstructionFlag.NONE;
        }

        @Override
        public <R, C> R accept(InsnVisitor<R, C> visitor, C ctx) {
            return visitor.visitSwitchSection(this, ctx);
        }

        //@formatter:off
        public Instruction getBody() { return body.get(); }
        public void setBody(Instruction body) { this.body.set(body); }
        //@formatter:on
    }

    public static class SwitchPattern extends Instruction {

        private final InstructionSlot<LocalReference> reference = new InstructionSlot<>(this);
        private final InstructionSlot<Instruction> condition = new InstructionSlot<>(this);

        public SwitchPattern(LocalReference reference) {
            this(reference, new Nop());
        }

        public SwitchPattern(LocalReference reference, Instruction condition) {
            this.reference.set(reference);
            this.condition.set(condition);
        }

        @Override
        public AType getResultType() {
            return PrimitiveType.VOID;
        }

        @Override
        public EnumBitSet<InstructionFlag> getDirectFlags() {
            return InstructionFlag.NONE;
        }

        @Override
        public <R, C> R accept(InsnVisitor<R, C> visitor, C ctx) {
            return visitor.visitSwitchPattern(this, ctx);
        }


        @Override
        protected void onChildModified() {
            super.onChildModified();
            if (isConnected()) {
                getReference().setWrittenTo(true);
            }
        }

        @Override
        protected void onConnected() {
            super.onConnected();
            getReference().setWrittenTo(true);
        }

        // @formatter:off
        public LocalReference getReference() { return reference.get(); }
        public Instruction getCondition() { return condition.get(); }
        // @formatter:on
    }

    public record TableInfo(int min, int max) { }
}
