package net.covers1624.coffeegrinder.bytecode.transform.transformers;

import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.matching.InvokeMatching;
import net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching;
import net.covers1624.coffeegrinder.bytecode.transform.ClassTransformContext;
import net.covers1624.coffeegrinder.bytecode.transform.ClassTransformer;
import net.covers1624.coffeegrinder.type.ClassType;
import net.covers1624.coffeegrinder.type.Field;
import net.covers1624.coffeegrinder.type.TypeSystem;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Objects;

import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.matchLoadLocal;
import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.matchStoreField;
import static net.covers1624.quack.util.SneakyUtils.notPossible;

/**
 * Created by covers1624 on 2/9/21.
 */
public class InnerClasses implements ClassTransformer {

    @SuppressWarnings ("NotNullFieldNotInitialized")
    private ClassType outerType;

    public void transform(ClassDecl cInsn, ClassTransformContext ctx) {
        cInsn.getClassMembers().forEach(e -> processInnerClass(e, ctx));
    }

    private void processInnerClass(ClassDecl cInsn, ClassTransformContext ctx) {
        if (cInsn.getClazz().isStatic()) return;

        ctx.pushStep("Remove synthetic Outer.this field");
        outerType = cInsn.getClazz().getEnclosingClass().orElseThrow(notPossible());

        FastStream<MethodDecl> constructors = cInsn.getMethodMembers().filter(f -> f.getMethod().isConstructor());

        Field thisField = constructors.map(this::matchOuterThisField).filter(Objects::nonNull).distinct().onlyOrDefault();
        assert thisField == null || ((ClassType) thisField.getType()).getDeclaration() == outerType;
        assert thisField == null || thisField.isSynthetic();

        constructors.forEach(this::removeOuterThisParam);

        if (thisField != null) {
            // LoadField(this$1)(LoadThis()) -> LoadThis(outer)
            cInsn.descendantsMatching(e -> LoadStoreMatching.matchLoadField(e, thisField))
                    .forEach(e -> e.replaceWith(new LoadThis(TypeSystem.makeThisType(outerType))));

            // remove the synthetic field
            cInsn.getFieldMembers().filter(f -> f.getField().equals(thisField)).only().remove();
        }

        ctx.popStep();
    }

    private void removeOuterThisParam(MethodDecl constructor) {
        // Remove the outer-this field, store, parameter usages, and the parameter.

        // Match
        // ...
        // STORE(FIELD this$1(LOAD_THIS(...)), LOAD arg0)
        // ...
        // <init>(Outer arg0, ...)
        // ->
        // ...
        // <init>(...)

        var thisParam = constructor.parameters.get(0);

        Store storeField = findOuterThisField(constructor, thisParam);
        if (storeField != null) storeField.remove();

        thisParam.makeImplicit();

        // there may be other references to the parameter (eg super constructor call)
        // super(Load arg0, ...) -> super(LoadThis(Outer), ...)
        List.copyOf(thisParam.getReferences())
                .forEach(e -> ((Load) e.getParent()).replaceWith(new LoadThis(TypeSystem.makeThisType(outerType))));
    }

    @Nullable
    private Field matchOuterThisField(MethodDecl constructor) {
        // Match the outer-this field for the ctor outer-this parameter.

        // Match:
        // ...
        // STORE(FIELD this$1(LOAD_THIS(...)), LOAD arg0)
        // ...
        // <init>(Outer arg0, ...)

        var thisParam = constructor.parameters.get(0);
        assert ((ClassType) thisParam.getType()).getDeclaration() == outerType;

        Store storeField = findOuterThisField(constructor, thisParam);
        if (storeField == null) return null;

        FieldReference fieldRef = (FieldReference) storeField.getReference();
        assert fieldRef.getTarget() instanceof LoadThis;
        assert LoadStoreMatching.matchLoadLocal(storeField.getValue(), thisParam) != null;
        return fieldRef.getField();
    }

    private @Nullable Store findOuterThisField(MethodDecl ctor, LocalVariable thisParam) {
        // Find the field store for the given parameter.

        // Match
        // ...
        // STORE(FIELD this$1(LOAD_THIS(...)), LOAD arg0)
        // <init>(Outer arg0, ...)

        // We start from the delegate call, it should be immediately above it.
        // J25 Flexible constructor bodies allow code to be placed between the ctor head
        // and the delegate call.
        var delegateCall = InvokeMatching.findDelegatedCtor(ctor);
        assert delegateCall != null;

        if (!(matchStoreField(delegateCall.getPrevSiblingOrNull()) instanceof Store store)) return null;
        if (matchLoadLocal(store.getValue(), thisParam) == null) return null;

        var fRef = ((FieldReference) store.getReference());
        assert fRef.getTarget() instanceof LoadThis;
        assert fRef.getField().isSynthetic();
        return store;
    }
}
