package net.covers1624.coffeegrinder.bytecode.transform.transformers;

import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.transform.BlockTransformContext;
import net.covers1624.coffeegrinder.bytecode.transform.BlockTransformer;
import net.covers1624.coffeegrinder.bytecode.transform.MethodTransformContext;
import net.covers1624.quack.collection.ColUtils;
import net.covers1624.quack.collection.FastStream;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import static java.util.Objects.requireNonNull;
import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.matchStoreLocal;

/**
 * Created by covers1624 on 8/2/24.
 */
public class SwitchExpressions implements BlockTransformer {

    @Override
    public void transform(Block block, BlockTransformContext ctx) {
        Instruction last = block.instructions.lastOrDefault();
        if (last instanceof Switch swtch) {
            transform(swtch, ctx);
        }
    }

    private void transform(Switch swtch, MethodTransformContext ctx) {
        record Candidate(Branch exit, Store store) { }

        var matches = swtch.descendantsOfType(Branch.class)
                .filter(e -> !e.getTargetBlock().isDescendantOf(swtch))
                .map(suspect -> {
                    // STORE(s_1, ...)
                    // BRANCH exit
                    if (!(matchStoreLocal(suspect.getPrevSiblingOrNull()) instanceof Store store)) return null;
                    if (store.getVariable().getKind() != LocalVariable.VariableKind.STACK_SLOT) return null;
                    return new Candidate(suspect, store);
                })
                .filter(Objects::nonNull) // TODO, FastStream.filterNonNull would be amazing. The nullability here is a mess.
                .toList();
        if (matches.isEmpty()) return;

        var firstMatch = matches.getFirst();
        var target = firstMatch.exit;
        var var = firstMatch.store.getVariable();
        // Var must be used once. In theory something could dup it, but we have no evidence
        // of javac duping it, we may need to re-evaluate this in the future should we find cases.
        if (var.getLoadCount() != 1) return;

        // Every exit must be the same, and must have the same var store.
        if (!ColUtils.allMatch(matches, c -> c.exit.getTargetBlock() == target.getTargetBlock() && c.store.getVariable() == var)) return;

        ctx.pushStep("Produce switch expression.");
        Switch newSwitch = new Switch(swtch.getValue(), swtch.getBody(), var.getType());
        for (var match : matches) {
            assert match != null;
            match.exit.replaceWith(new Yield(newSwitch, match.store.getValue()).withOffsets(match.exit));
            match.store.remove();
        }
        var store = swtch.replaceWith(new Store(new LocalReference(var), newSwitch));
        store.insertAfter(new Branch(target.getTargetBlock()));
        ctx.popStep();
    }
}
