package net.covers1624.coffeegrinder.debug;

import net.covers1624.coffeegrinder.bytecode.DebugPrintOptions;
import net.covers1624.coffeegrinder.debug.Step.StepContextType;
import net.covers1624.coffeegrinder.debug.Step.StepType;
import net.covers1624.quack.util.SneakyUtils;
import org.jetbrains.annotations.Nullable;

import java.util.LinkedList;
import java.util.function.Supplier;

/**
 * A content tracing {@link Stepper} implementation.
 * <p>
 * Created by covers1624 on 13/5/21.
 */
public class DebugStepper implements Stepper {

    private final DebugPrintOptions opts;
    private final int stopAtStep;
    private final LinkedList<Stepper.ContextSupplier> contextStack = new LinkedList<>();

    private int nextStepId;

    @Nullable
    private Step root = null;

    @Nullable
    private Throwable exception;

    private final LinkedList<Step> steps = new LinkedList<>();

    public DebugStepper(DebugPrintOptions opts, int stopAtStep) {
        this.opts = opts;
        this.stopAtStep = stopAtStep;
    }

    @Override
    public DebugPrintOptions getOpts() {
        return opts;
    }

    @Override
    public void pushContext(Stepper.ContextSupplier supplier) {
        contextStack.push(supplier);
    }

    @Override
    public void popContext() {
        contextStack.pop();
    }

    @Override
    public Step pushStep(String name) {
        return step(StepType.CONTENT, StepContextType.NONE, name, null);
    }

    @Override
    public Step pushStep(String name, StepContextType contextType) {
        return step(StepType.CONTENT, contextType, name, null);
    }

    @Override
    public Step pushStepWithContent(String name, StepContextType contextType, Supplier<String> preContent) {
        return step(StepType.CONTENT, contextType, name, preContent.get());
    }

    @Override
    public void popStep() {
        Step step = steps.pop();
        if (step.finish()) {
            insertPostMicroStep(step);
        }
    }

    @Override
    public Step pushTiming(String name) {
        return step(StepType.TIMING, StepContextType.NONE, name, null);
    }

    private Step step(StepType type, StepContextType contextType, String name, @Nullable String preContent) {
        Step parent = steps.peek();

        Stepper.ContextSupplier supplier = null;
        if (type != StepType.TIMING) {
            supplier = contextStack.peek();
            assert supplier != null : "Attempted to step without context.";

            insertPushMicroStep(parent, supplier);
        }

        Step step = new Step(this, nextStepId(), type, contextType, name, parent, preContent, supplier);
        steps.push(step);
        if (parent != null) {
            assert !parent.isFinished;
            parent.children.add(step);
        } else {
            root = step;
        }
        return step;
    }

    private void insertPushMicroStep(@Nullable Step parent, Stepper.ContextSupplier supplier) {
        if (parent == null) return;
        Step prevStep = !parent.children.isEmpty() ? parent.children.peekLast() : parent;
        if (prevStep.supplier != supplier) return;

        String prev = prevStep == parent ? prevStep.getPreStepContent() : prevStep.getPostStepContent();
        long start = prevStep == parent ? prevStep.getStartTime() : prevStep.getEndTime();
        if (prev == null) return;

        String content;
        try {
            content = supplier.print(opts);
        } catch (Throwable ignored) {
            return;
        }
        if (prev.equals(content)) return;

        parent.children.add(new Step(this, nextStepId(), "...", parent, prev, content, start, System.nanoTime()));
    }

    private void insertPostMicroStep(Step parent) {
        if (parent.children.isEmpty()) return;

        Step lastChild = parent.children.getLast();
        if (lastChild.getType() == StepType.TIMING) return;
        if (parent.getPostStepContent() == null) return;
        if (parent.getStatus() == Step.Status.ERROR) return;
        String prev = lastChild.getPostStepContent();
        if (prev == null) return;
        if (parent.getPostStepContent().equals(lastChild.getPostStepContent())) return;

        parent.children.add(new Step(this, nextStepId(), "...", parent, prev, parent.getPostStepContent(), lastChild.getEndTime(), parent.getEndTime()));
    }

    @Override
    public void popTiming() {
        steps.pop().finish();
    }

    @Override
    public void except(Throwable e) {
        assert exception == null;
        exception = e;
        SneakyUtils.throwUnchecked(e);
    }

    @Override
    @Nullable
    public Step getRoot() {
        return root;
    }

    @Nullable
    Throwable clearExcept() {
        Throwable e = exception;
        exception = null;
        return e;
    }

    /**
     * Hello and welcome.
     *
     * Please step-out a few functions in your debugger.
     */
    int nextStepId() {
        int id = nextStepId++;
        if (id == stopAtStep) {
            try {
                throw new DebugStopAtStep();
            } catch (DebugStopAtStep ignored) {
            }
        }
        return id;
    }
}
