package net.covers1624.coffeegrinder.bytecode.transform.transformers;

import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.ScopeVisitor;
import net.covers1624.coffeegrinder.bytecode.SimpleInsnVisitor;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.transform.MethodTransformContext;
import net.covers1624.coffeegrinder.bytecode.transform.MethodTransformer;
import net.covers1624.coffeegrinder.util.None;

import java.util.*;

import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.matchLocalRef;
import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.matchStoreLocal;

/**
 * A variable is in scope, if the parent of the declaration (IStoreInstruction) is an ancestor of the usage
 * and the declaration has a lower child index in the common ancestor than the usage
 * <p>
 * Split variables with an aliased declaration in-scope are merged.
 * Created by covers1624 on 15/9/21.
 */
public class VariableDeclarations extends SimpleInsnVisitor<None> implements MethodTransformer {

    private final Map<LocalVariable, Instruction> varDeclPoints = new LinkedHashMap<>();

    @Override
    public void transform(MethodDecl function, MethodTransformContext ctx) {
        varDeclPoints.clear();
        function.accept(this);

        for (Map.Entry<LocalVariable, Instruction> entry : varDeclPoints.entrySet()) {
            LocalVariable var = entry.getKey();
            Instruction declPoint = entry.getValue();

            // if we have chosen another instruction as our decl point, make a bare reference as an explicit declaration
            if (!isDeclaration(declPoint, var)) {
                ctx.pushStep("Declare " + var.getUniqueName() + " in outer scope");
                LocalReference newDecl = new LocalReference(var);
                declPoint.insertBefore(newDecl);
                ctx.popStep();
            }
        }

        function.accept(new MergeAliasedDeclarations(), ctx);
    }

    private boolean isDeclaration(Instruction declPoint, LocalVariable var) {
        return matchLocalRef(declPoint, var) != null || matchStoreLocal(declPoint, var) != null;
    }

    private boolean replaceVariable(LocalVariable var, LocalVariable replacement, MethodTransformContext ctx) {
        // when decompiling without a LV table, we may want to see if we can merge the types of the variable. But that does require checking compliance with all usages...
        if (!var.getType().equals(replacement.getType()) || var.getIndex() != replacement.getIndex()) return false;
        ctx.pushStep("Merge variable " + var.getUniqueName() + " into " + replacement.getUniqueName());
        List.copyOf(var.getReferences()).forEach(r -> replace(r, replacement));
        ctx.popStep();
        return true;
    }

    private void replace(LocalReference r, LocalVariable replacement) {
        if (r.isWrittenTo() || r.isReadFrom()) {
            r.replaceWith(new LocalReference(replacement));
        } else {
            r.remove();
        }
    }

    private static boolean isImmovableDecl(Instruction decl) {
        return switch (decl.getParent()) {
            case InstanceOf instanceOf -> instanceOf.getPattern() == decl;
            case SwitchTable.SwitchPattern pattern -> pattern.getReference() == decl;
            case TryCatch tryCatch -> tryCatch.resources.anyMatch(e -> e == decl);
            case TryCatch.TryCatchHandler handler -> handler.getVariable() == decl;
            case ForEachLoop forEachLoop -> forEachLoop.getVariable() == decl;
            default -> false;
        };
    }

    /**
     * Finds a suitable declaration location such that usage is in scope with the previous declaration.
     * Assumes that decl is 'before' usage
     *
     * @param decl  The previous declaration for a variable
     * @param usage An instruction which requires the declaration to be in scope
     * @return <code>decl</code>, if <code>decl</code> is in scope, otherwise a common ancestor of both <code>decl</code> and <code>usage</code>,
     * which is a child of a block (so a declaration could be inserted before it)
     */
    public static Instruction unifyUsages(Instruction decl, Instruction usage) {
        if (isImmovableDecl(decl)) return decl;

        Instruction commonParent = findCommonParent(decl, usage);

        if (commonParent instanceof ForLoop loop && loop.getInitializer() == decl) {
            return decl;
        }

        if (commonParent instanceof Block) { // select the current declaration within the common parent block
            return findAncestorChild(decl, commonParent);
        }

        // common parent may be a single instruction (eg an if)
        return selectDeclarableParent(commonParent);
    }

    /**
     * find the first declaration point, the first ancestor which is a child of a block (which may be this instruction, or a parent)
     */
    public static Instruction selectDeclarableParent(Instruction insn) {
        while (!(insn.getParent() instanceof Block block) || block.getFirstChildOrNull() instanceof SwitchTable) {
            insn = insn.getParent();

            // an escape hatch for when the root container has multiple blocks. Indicates previous transforms have failed to remove all gotos
            if (insn instanceof MethodDecl mDecl) {
                // declare at the top of the function. Hope for the best
                return mDecl.getBody().getEntryPoint().getFirstChild();
            }
        }
        return insn;
    }

    private static Instruction findAncestorChild(Instruction insn, Instruction ancestor) {
        while (insn.getParent() != ancestor) {
            insn = insn.getParent();
        }

        return insn;
    }

    private static Instruction findCommonParent(Instruction decl, Instruction use2) {
        Instruction parent = decl;
        do {
            parent = parent.getParent();
        }
        while (!use2.isDescendantOf(parent));
        return parent;
    }

    @Override
    public None visitLocalReference(LocalReference localRef, None ctx) {
        LocalVariable var = localRef.variable;
        if (var.getKind() == LocalVariable.VariableKind.PARAMETER) return NONE;

        Instruction scope = localRef.getParent() instanceof Store ? localRef.getParent() : localRef;
        Instruction decl = varDeclPoints.get(var);
        if (decl == null) {
            assert localRef.isWrittenTo();
            varDeclPoints.put(var, scope);
        } else {
            varDeclPoints.put(var, unifyUsages(decl, scope));
        }

        return NONE;
    }

    private class MergeAliasedDeclarations extends SimpleInsnVisitor<MethodTransformContext> {

        private final ScopeVisitor<None, MethodTransformContext> scopeVisitor = new ScopeVisitor<>(this);
        private final Set<LocalVariable> seenVars = new HashSet<>();

        @Override
        public None visitDefault(Instruction insn, MethodTransformContext ctx) {
            return visitChildren(scopeVisitor::visit, insn, ctx);
        }

        @Override
        public None visitSwitchSection(SwitchTable.SwitchSection switchSection, MethodTransformContext ctx) {
            super.visitSwitchSection(switchSection, ctx);
            switchSection.explicitBlock = scopeVisitor.currentScopeHasDeclarations();
            return NONE;
        }

        @Override
        public None visitLocalReference(LocalReference local, MethodTransformContext ctx) {
            if (!seenVars.add(local.variable)) return NONE;

            var declared = scopeVisitor.getVariableInScope(local.variable.getName());
            if (declared != null && declared != local.variable && !isImmovableDecl(local)) {
                replaceVariable(local.variable, declared, ctx);
            }

            return NONE;
        }
    }
}
