package net.covers1624.coffeegrinder.bytecode.insns;

import net.covers1624.coffeegrinder.bytecode.*;
import net.covers1624.coffeegrinder.type.AType;
import net.covers1624.coffeegrinder.type.PrimitiveType;
import net.covers1624.coffeegrinder.util.EnumBitSet;
import org.jetbrains.annotations.Nullable;

/**
 * Created by covers1624 on 28/5/21.
 */
public final class TryCatch extends Instruction {

    public final InstructionCollection<Instruction> resources = new InstructionCollection<>(this);
    public final InstructionSlot<BlockContainer> tryBody = new InstructionSlot<>(this);
    public final InstructionCollection<TryCatchHandler> handlers = new InstructionCollection<>(this);
    private final InstructionSlot<Instruction> finallyBody = new InstructionSlot<>(this);

    public TryCatch(BlockContainer tryBody) {
        this.tryBody.set(tryBody);
        this.finallyBody.set(new Nop());
    }

    @Override
    public AType getResultType() {
        return PrimitiveType.VOID;
    }

    @Override
    protected EnumBitSet<InstructionFlag> computeFlags() {
        EnumBitSet<InstructionFlag> flags = getTryBody().getFlags().copy();
        for (TryCatchHandler handler : handlers) {
            flags = SemanticHelper.combineBranches(flags, handler.getFlags());
        }

        // If the finally is END_POINT_UNREACHABLE then the body/catches never exit normally.
        if (getFinallyBody() instanceof BlockContainer finBody && finBody.getFlags().get(InstructionFlag.END_POINT_UNREACHABLE)) {
            flags.set(InstructionFlag.END_POINT_UNREACHABLE);
        }

        return flags;
    }

    @Override
    public EnumBitSet<InstructionFlag> getDirectFlags() {
        return InstructionFlag.NONE;
    }

    @Override
    public <R, C> R accept(InsnVisitor<R, C> visitor, C ctx) {
        return visitor.visitTryCatch(this, ctx);
    }

    //@formatter:off
    public BlockContainer getTryBody() { return tryBody.get(); }
    public void setTryBody(BlockContainer tryBody) { this.tryBody.set(tryBody); }
    @Nullable public BlockContainer getFinallyBody() { return finallyBody.get() instanceof BlockContainer cont ? cont : null; }
    public void setFinallyBody(@Nullable BlockContainer finallyBody) { this.finallyBody.set(finallyBody != null ? finallyBody : new Nop()); }
    //@formatter:on

    public static final class TryCatchHandler extends Instruction {

        private final InstructionSlot<LocalReference> variable = new InstructionSlot<>(this);
        private final InstructionSlot<BlockContainer> body = new InstructionSlot<>(this);

        public boolean isUnprocessedFinally = false;

        public TryCatchHandler(BlockContainer body, LocalReference variable) {
            this.variable.set(variable);
            this.body.set(body);
        }

        @Override
        public AType getResultType() {
            return PrimitiveType.VOID;
        }

        @Override
        public EnumBitSet<InstructionFlag> getDirectFlags() {
            return InstructionFlag.NONE;
        }

        @Override
        public <R, C> R accept(InsnVisitor<R, C> visitor, C ctx) {
            return visitor.visitTryCatchHandler(this, ctx);
        }

        @Override
        protected void onChildModified() {
            super.onChildModified();
            if (isConnected()) {
                getVariable().setWrittenTo(true);
            }
        }

        @Override
        protected void onConnected() {
            super.onConnected();
            getVariable().setWrittenTo(true);
        }

        //@formatter:off
        public LocalReference getVariable() { return variable.get(); }
        public BlockContainer getBody() { return body.get(); }
        public void setVariable(Store variable) { this.variable.set(variable); }
        public TryCatch getTry() { return ((TryCatch) getParent()); }
        //@formatter:on
    }
}
