package net.covers1624.coffeegrinder.bytecode.transform.transformers;

import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.insns.tags.RecordPatternComponentTag;
import net.covers1624.coffeegrinder.bytecode.transform.MethodTransformContext;
import net.covers1624.coffeegrinder.type.ClassType;
import net.covers1624.coffeegrinder.util.Util;
import net.covers1624.quack.collection.ColUtils;
import net.covers1624.quack.collection.FastStream;

import java.util.ArrayList;
import java.util.Objects;

import static net.covers1624.coffeegrinder.bytecode.insns.Invoke.InvokeKind.VIRTUAL;
import static net.covers1624.coffeegrinder.bytecode.matching.InvokeMatching.matchInvoke;
import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.matchLoadLocal;

/**
 * Created by covers1624 on 12/22/25.
 */
public class RecordPatterns {

    public static boolean tryMakeRecordPattern(LocalReference existingPattern, MethodTransformContext ctx) {
        // Var must be a record.
        if (!(existingPattern.getType() instanceof ClassType type) || !type.isRecord()) return false;

        var components = type.getRecordComponents();
        var matchedPatterns = new ArrayList<MatchedPattern>();
        for (int i = 0; i < components.size(); i++) {
            matchedPatterns.add(null);
        }

        // Try and match each variable to one of the following patterns:
        // STORE var = [RecordPatternTag]INVOKE.VIRTUAL TheRecord.component(LOAD reference)
        // OR
        // [RecordPatternTag]INVOKE.VIRTUAL TheRecord.component(LOAD reference) instanceof ... nested_pattern
        var storeVar = existingPattern.variable;
        for (var reference : storeVar.getReferences()) {
            if (reference == existingPattern) continue;

            if (!(matchLoadLocal(reference.getParent()) instanceof Load refLoad)) return false;
            if (!(matchInvoke(refLoad.getParent(), VIRTUAL) instanceof Invoke retriever)) return false;
            if (!(retriever.getTag() instanceof RecordPatternComponentTag)) return false;
            var name = retriever.getMethod().getName();

            var index = Util.indexOf(components, e -> e.getName().equals(name));
            if (index == -1) return false;
            assert matchedPatterns.get(index) == null;

            matchedPatterns.set(index, switch (retriever.getParent()) {
                case Store usageStore when usageStore.getReference() instanceof LocalReference -> new VariableCapturePattern(usageStore);
                case InstanceOf patt when !(patt.getPattern() instanceof Nop) -> new InvokeCapturePattern(retriever);
                default -> null;
            });

        }

        // If we didn't find a pattern for each var usage.
        if (ColUtils.anyMatch(matchedPatterns, Objects::isNull)) return false;

        ctx.pushStep("Create record pattern");
        var func = existingPattern.firstAncestorOfType(MethodDecl.class);
        // TODO replace Nop with ReplacementPlaceholder, which prints as a `?` in both printers.
        var pattern = new RecordPattern(type, FastStream.of(components).map(e -> new Nop()));
        existingPattern.replaceWith(pattern);
        for (int i = 0; i < matchedPatterns.size(); i++) {
            MatchedPattern o = matchedPatterns.get(i);
            var name = components.get(i).getName();

            switch (o) {
                // Variable usage used for local variable.
                // STORE var = [RecordPatternTag]INVOKE.VIRTUAL TheRecord.component(LOAD reference)
                // Directly capture 'var' reference.
                case VariableCapturePattern(Store usage) -> {
                    ctx.pushStep("Local " + usage.getVariable().getName() + " for " + name);
                    pattern.args.get(i).replaceWith(usage.getReference());
                    usage.remove();
                    ctx.popStep();
                }
                // Variable usage used for nested pattern.
                // [RecordPatternTag]INVOKE.VIRTUAL TheRecord.component(LOAD reference) instanceof ... nested_pattern
                // In order to make things easier, we opt to replace the invoke with a temporary variable
                // we can then pick the nested instanceof into its main pattern later after we have produced short circuits.
                case InvokeCapturePattern(Invoke invoke) -> {
                    ctx.pushStep("Temp local for nested pattern " + name);
                    var variable = new LocalVariable(LocalVariable.VariableKind.TEMP_LOCAL, invoke.getResultType(), storeVar.getName() + "$" + invoke.getMethod().getName());
                    func.variables.add(variable);
                    invoke.replaceWith(new Load(new LocalReference(variable)).withTag(invoke.getTag()));
                    pattern.args.get(i).replaceWith(new LocalReference(variable));
                    ctx.popStep();
                }
            }
        }

        ctx.popStep();
        return true;
    }

    private sealed interface MatchedPattern permits VariableCapturePattern, InvokeCapturePattern { }

    private record VariableCapturePattern(Store store) implements MatchedPattern { }

    private record InvokeCapturePattern(Invoke invoke) implements MatchedPattern { }
}
