package net.covers1624.coffeegrinder.type;

import net.covers1624.coffeegrinder.bytecode.AccessFlag;
import net.covers1624.coffeegrinder.util.EnumBitSet;
import net.covers1624.coffeegrinder.util.Util;
import net.covers1624.quack.collection.FastStream;
import net.covers1624.quack.util.JavaVersion;
import org.jetbrains.annotations.Nullable;
import org.objectweb.asm.Type;

import java.lang.annotation.ElementType;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Supplier;

import static net.covers1624.quack.collection.FastStream.of;

/**
 * Created by covers1624 on 21/2/22.
 */
public abstract class ClassType extends ReferenceType implements ITypeParameterizedMember {

    protected final Supplier<Map<String, Field>> fieldsLookup = Util.singleMemoize(() -> FastStream.of(getFields())
            .toImmutableMap(e -> e.getName() + e.getDescriptor(), e -> e));

    protected final Supplier<Map<String, Method>> methodsLookup = Util.singleMemoize(() -> FastStream.of(getMethods())
            .toImmutableMap(e -> e.getName() + e.getDescriptor(), e -> e));

    public abstract ClassType getSuperClass();

    public abstract List<ClassType> getInterfaces();

    public abstract List<ClassType> getNestedClasses();

    public abstract List<Field> getFields();

    public abstract List<Method> getMethods();

    public abstract DeclType getDeclType();

    public abstract String getPackage();

    public abstract EnumBitSet<AccessFlag> getAccessFlags();

    public abstract Optional<ClassType> getEnclosingClass();

    public abstract Optional<Method> getEnclosingMethod();

    public AnnotationSupplier getAnnotationSupplier() {
        throw new UnsupportedOperationException();
    }

    public List<ElementType> getAnnotationTargets() {
        throw new UnsupportedOperationException();
    }

    public abstract Type getDescriptor();

    public abstract ClassType getDeclaration();

    public abstract ClassType asRaw();

    public JavaVersion getClassVersion() {
        return getDeclaration().getClassVersion();
    }

    public final boolean isInterface() {
        return getAccessFlags().get(AccessFlag.INTERFACE);
    }

    public final boolean isStatic() {
        return getAccessFlags().get(AccessFlag.STATIC);
    }

    public final boolean isFinal() {
        return getAccessFlags().get(AccessFlag.FINAL);
    }

    public final boolean isEnum() {
        return getAccessFlags().get(AccessFlag.ENUM);
    }

    public final boolean isRecord() {
        return getAccessFlags().get(AccessFlag.RECORD);
    }

    public final boolean isSynthetic() {
        return getAccessFlags().get(AccessFlag.SYNTHETIC);
    }

    public List<ClassType> getPermittedSubclasses() {
        return getDeclaration().getPermittedSubclasses();
    }

    // TODO we have gotten this far without actually using the RecordComponents attribute, perhaps we should use it anyway.
    public List<Field> getRecordComponents() {
        assert isRecord();
        return FastStream.of(getFields())
                .filter(e -> !e.isStatic())
                .toList();
    }

    public TypeResolver getTypeResolver() {
        return getDeclaration().getTypeResolver();
    }

    @Nullable
    @Override
    public TypeParameter resolveTypeParameter(String identifier) {
        // Try ourselves first.
        for (TypeParameter param : getTypeParameters()) {
            if (param.getName().equals(identifier)) {
                return param;
            }
        }
        // Otherwise, our enclosing method.
        if (getEnclosingMethod().isPresent()) {
            return getEnclosingMethod().get().resolveTypeParameter(identifier);
        }

        // Other, Otherwise, our declaring class.
        if (getEnclosingClass().isPresent()) {
            return getEnclosingClass().get().resolveTypeParameter(identifier);
        }
        return null;
    }

    @Override
    public ReferenceType getSuperType() {
        return getSuperClass();
    }

    @Override
    public FastStream<ClassType> getDirectSuperTypes() {
        if (TypeSystem.isObject(this)) return FastStream.empty();

        return of(getSuperClass()).concat(getInterfaces());
    }

    @Nullable
    public Field resolveField(String name, Type desc) {
        Field field = fieldsLookup.get().get(name + desc);
        if (field != null) return field;

        return super.resolveField(name, desc);
    }

    @Nullable
    public Method resolveMethod(String name, Type desc) {
        Method method = methodsLookup.get().get(name + desc);
        if (method != null && !method.isBridge()) return method;

        return super.resolveMethod(name, desc);
    }

    @Nullable
    public Field findConstant(Object value, boolean isStatic) {
        return getDeclaration().findConstant(value, isStatic);
    }

    @Nullable
    @Override
    public Method getFunctionalInterfaceMethod() {
        if (!isInterface()) {
            throw new IllegalArgumentException("Not an interface");
        }

        // TODO: Cache this.
        Method[] options = FastStream.of(getAllMethods())
                .filter(Method::isAbstract)
                .filter(m -> getSuperClass().resolveMethod(m.getName(), m.getDescriptor()) == null)
                .toArray(new Method[0]);
        if (options.length == 0) return null;

        if (options.length == 1) return options[0];

        // https://docs.oracle.com/javase/specs/jls/se17/html/jls-9.html#jls-9.9
        Method m = selectMethodWithSubsignatureOfAll(options);

        // todo, if m is null, manufacture a method signature
        return m;
    }

    @Nullable
    private static Method selectMethodWithSubsignatureOfAll(Method[] options) {
        // technically, the signature is the lub of all the params of all the candidates
        // then if a method matches that signature, send it
        // proper impl would be to manufacture a signature and then check methods against it
        outer:
        for (int i = 0; i < options.length; i++) {
            Method m = options[i];
            for (int j = 0; j < options.length; j++) {
                if (j != i && !isSubsignatureOf(m, options[j])) {
                    continue outer;
                }
            }
            return m;
        }

        return null;
    }

    private static boolean isSubsignatureOf(Method s, Method t) {
        // todo, generics
        for (int i = 0; i < s.getParameters().size(); i++) {
            AType paramS = s.getParameters().get(i).getType();
            AType paramT = t.getParameters().get(i).getType();
            if (!(paramS instanceof ReferenceType)) continue;
            if (!TypeSystem.lub((ReferenceType) paramS, (ReferenceType) paramT).equals(paramS)) {
                return false;
            }
        }

        AType retS = s.getReturnType();
        AType retT = t.getReturnType();
        return TypeSystem.isAssignableTo(retS, retT);
    }

    @Override
    public boolean mentions(ReferenceType type) {
        return super.mentions(type) || getDeclaration() == type;
    }

    public ClassType getTopLevelClass() {
        var type = getEnclosingClass()
                .map(ClassType::getTopLevelClass)
                .orElse(this);
        assert type.getDeclType() == DeclType.TOP_LEVEL;
        return type;
    }

    @Override
    public String toString() {
        return getFullName();
    }

    public enum DeclType {
        TOP_LEVEL,
        INNER,
        LOCAL,
        ANONYMOUS;

        public boolean isLocalOrAnonymous() {
            return this == LOCAL || this == ANONYMOUS;
        }
    }
}
