package net.covers1624.coffeegrinder.source;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by covers1624 on 8/8/21.
 */
public class LineBuffer {

    public static final LineBuffer EMPTY = of(List.of());

    public final List<String> lines;

    private LineBuffer(List<String> lines) {
        this.lines = List.copyOf(lines); // Does not copy if input is an ImmutableList.
    }

    public static LineBuffer of() {
        return EMPTY;
    }

    public static LineBuffer of(Object str) {
        return of(List.of(str.toString()));
    }

    public static LineBuffer paren(LineBuffer buffer) {
        return of("(").append(buffer).append(")");
    }

    public static LineBuffer of(List<String> lines) {
        return new LineBuffer(lines);
    }

    public LineBuffer prepend(String prepend) {
        if (prepend.isEmpty()) return this;
        if (lines.size() == 1) return of(prepend + lines.getFirst());

        List<String> builder = new ArrayList<>(lines.size());
        builder.add(prepend + lines.getFirst());
        builder.addAll(lines.subList(1, lines.size()));
        return of(builder);
    }

    public LineBuffer append(Object append) {
        String str = append.toString();
        if (lines.isEmpty()) return of(List.of(str));
        if (lines.size() == 1) return of(lines.getFirst() + str);
        if (str.isEmpty()) return this;

        List<String> builder = new ArrayList<>(lines.size());
        builder.addAll(lines.subList(0, lines.size() - 1));
        builder.add(lines.getLast() + str);
        return of(builder);
    }

    public LineBuffer append(LineBuffer other) {
        return append("", other);
    }

    public LineBuffer append(String sep, LineBuffer other) {
        if (lines.isEmpty()) return other.prepend(sep);
        if (other.lines.isEmpty()) return append(sep);

        List<String> builder = new ArrayList<>(lines.size() + other.lines.size());

        builder.addAll(lines.subList(0, lines.size() - 1));
        builder.add(lines.getLast() + sep + other.lines.getFirst());
        builder.addAll(other.lines.subList(1, other.lines.size()));
        return of(builder);
    }

    public LineBuffer add(String line) {
        List<String> builder = new ArrayList<>(lines.size() + 1);
        builder.addAll(lines);
        builder.add(line);
        return of(builder);
    }

    public LineBuffer join(LineBuffer other) {
        if (lines.isEmpty()) return other;
        if (other.lines.isEmpty()) return this;

        List<String> builder = new ArrayList<>(lines.size() + other.lines.size());
        builder.addAll(lines);
        builder.addAll(other.lines);
        return of(builder);
    }

    public LineBuffer joinOn(String seperator) {
        return of(String.join(seperator, lines));
    }

    @Override
    public String toString() {
        return String.join("\n", lines);
    }

    public LineBuffer stripStart() {
        if (lines.isEmpty()) return this;

        List<String> lines = new ArrayList<>(this.lines);
        lines.set(0, lines.getFirst().stripLeading());
        return of(lines);
    }
}
