package net.covers1624.coffeegrinder.type;

import net.covers1624.quack.collection.ColUtils;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;

import java.util.*;

import static java.util.Objects.requireNonNull;
import static net.covers1624.coffeegrinder.type.PrimitiveType.*;
import static net.covers1624.quack.util.SneakyUtils.notPossible;
import static net.covers1624.quack.util.SneakyUtils.unsafeCast;

/**
 * Created by covers1624 on 15/8/21.
 */
public class TypeSystem {

    private TypeSystem() { } // No!

    /**
     * @param type The type to search the hierarchy of
     * @param decl The parent type
     * @return true if {@code decl} exists on the hierarchy of (or equals) {@code type}
     */
    public static boolean isSubType(ReferenceType type, ReferenceType decl) {
        if (getDecl(type).equals(decl)) return true;
        return type.getDirectSuperTypes().anyMatch(e -> isSubType(e, decl));
    }

    // region isAssignableTo
    public static boolean isAssignableTo(AType from, AType to) {
        return isAssignableTo(from, to, true);
    }

    public static boolean isAssignableTo(AType from, AType to, boolean allowUnchecked) {
        if (from instanceof ReferenceType) {
            return to instanceof ReferenceType && isAssignableTo((ReferenceType) from, (ReferenceType) to, allowUnchecked);
        }

        return isAssignableToPrimitive(from, to);
    }

    private static boolean isAssignableToPrimitive(AType from, AType to) {

        if (from.equals(to)) return true;

        if (from instanceof IntegerConstantUnion) { //allMatch isAssignableTo
            for (AType type : ((IntegerConstantUnion) from).getTypes()) {
                if (!isAssignableToPrimitive(type, to)) return false;
            }
            return true;
        }

        if (to instanceof IntegerConstantUnion) {
            return FastStream.of(((IntegerConstantUnion) to).getTypes()).anyMatch(t -> isAssignableToPrimitive(from, t));
        }

        if (from instanceof IntegerConstantType) {
            int value = ((IntegerConstantType) from).getValue();
            if (to == INT || to == LONG || to == FLOAT || to == DOUBLE) return true;
            if (Short.MIN_VALUE <= value && value <= Short.MAX_VALUE && to == SHORT) return true;
            if (Character.MIN_VALUE <= value && value <= Character.MAX_VALUE && to == CHAR) return true;
            return Byte.MIN_VALUE <= value && value <= Byte.MAX_VALUE && to == BYTE;
        }

        if (from instanceof PrimitiveType) {
            if (from == VOID) {
                throw new UnsupportedOperationException("VOID should not even be in a place where you're querying implicit convertibility");
            }

            if (!(to instanceof PrimitiveType)) return false;
            if (from == FLOAT) return to == DOUBLE;
            boolean isFloatTarget = to == FLOAT || to == DOUBLE;
            if (from == INT) return to == LONG || isFloatTarget;
            if (from == SHORT || from == CHAR) return to == INT || to == LONG || isFloatTarget;
            if (from == BYTE) return to == SHORT || to == INT || to == LONG || isFloatTarget;
            if (from == LONG) return isFloatTarget;
            return false;
        }

        throw new UnsupportedOperationException("Assigning " + from + " to " + to);
    }

    public static boolean isAssignableTo(ReferenceType from, ReferenceType to) {
        return isAssignableTo(from, to, true);
    }

    public static boolean isAssignableTo(ReferenceType from, ReferenceType to, boolean allowUnchecked) {
        assert isFullyDefined(from);
        assert isFullyDefined(to);

        if (from == NullConstantType.INSTANCE) return true;
        if (to == NullConstantType.INSTANCE) return false;

        if (from.equals(to)) return true;
        if (isObject(to)) return true;
        if (isObject(from)) return false;

        assert !(to instanceof ReferenceUnionType) : "Union types are final. JLS 14.20";

        if (to instanceof ArrayType && from instanceof ArrayType) {
            return isAssignableTo(((ArrayType) from).getElementType(), ((ArrayType) to).getElementType(), allowUnchecked);
        }

        ReferenceType lower = to.getLowerBound();
        if (lower != to && isAssignableTo(from, lower, allowUnchecked)) return true;

        if (to instanceof IntersectionType) return to.getDirectSuperTypes().allMatch(e -> isAssignableTo(from, e, allowUnchecked));

        if (from instanceof ClassType) {
            if (!(to instanceof ClassType)) return false;

            if (((ClassType) from).getDeclaration() == ((ClassType) to).getDeclaration()) { // make a decision here, no point going further up the hierarchy
                if (to instanceof ParameterizedClass) {
                    if (from instanceof ParameterizedClass) {
                        return areParameterizationsAssignable((ParameterizedClass) from, (ParameterizedClass) to);
                    }

                    return allowUnchecked;
                }

                // assigning to raw is always permitted, though raw types should be avoided
                return true;
            }
        }

        return from.getDirectSuperTypes().anyMatch(e -> isAssignableTo(e, to, allowUnchecked));
    }

    private static boolean areParameterizationsAssignable(ParameterizedClass pFrom, ParameterizedClass pTo) {
        List<ReferenceType> fromArgs = pFrom.getTypeArguments();
        List<ReferenceType> toArgs = pTo.getTypeArguments();
        for (int i = 0; i < fromArgs.size(); i++) {
            ReferenceType from = fromArgs.get(i);
            ReferenceType to = toArgs.get(i);
            if (!isContainedBy(from, to)) {
                return false;
            }
        }

        if (pFrom.getOuter() != null) return areParameterizationsAssignable(pFrom.getOuter(), requireNonNull(pTo.getOuter()));

        return true;
    }

    private static boolean isContainedBy(ReferenceType a, ReferenceType b) {
        // a.getUpperBound <= b.getUpperBound && a.getLowerBound >= b.getLowerBound
        if (a.equals(b)) return true;

        // must be able to accept the lower bound of b
        if (!isAssignableTo(b.getLowerBound(), a.getLowerBound(), false)) {
            return false;
        }

        // only use the upper bounds if the types are wildcards. TypeParams are treated as actual types here
        // in normal assignability, a cannot be a wildcard, only a capture, but since we use isContainedBy for subtyping checks as well, we need to support it
        ReferenceType fromUpper = a instanceof WildcardType ? a.getUpperBound() : a;
        ReferenceType toUpper = b instanceof WildcardType ? b.getUpperBound() : b;
        return isAssignableTo(fromUpper, toUpper, b instanceof WildcardType);
    }
    // endregion

    // region isCastableTo
    // https://docs.oracle.com/javase/specs/jls/se9/html/jls-5.html#jls-5.1.6.1
    // allowWidening should always be true for now, but longer term, the Cast invariant should prevent it at some point, as it's redundant
    public static boolean isCastableTo(ReferenceType from, ReferenceType to, boolean allowWidening) {
        if (isAssignableTo(from, to)) return allowWidening; // Redundant cast.

        if (from instanceof ArrayType && to instanceof ArrayType) {
            AType fromElem = ((ArrayType) from).getElementType();
            AType toElem = ((ArrayType) to).getElementType();
            if (fromElem instanceof ReferenceType && toElem instanceof ReferenceType) {
                return isCastableTo((ReferenceType) fromElem, (ReferenceType) toElem, false);
            }
            return false; // only exact/widening casts are allowed for primitive arrays
        }

        // probably part of the JLS somewhere?
        if (from instanceof WildcardType) return isCastableTo(from.getUpperBound(), to, false);
        if (to instanceof WildcardType) return false; // can never cast to a wildcard, only assign (and even then, only to a capture)
        if (to instanceof CapturedTypeVar) return false;

        if (from instanceof TypeVariable) return isCastableTo(from.getUpperBound(), to, false);
        if (to instanceof TypeParameter) return isCastableTo(from, to.getUpperBound(), true);

        if (from instanceof IntersectionType) return from.getDirectSuperTypes().allMatch(e -> isCastableTo(e, to, true));
        if (to instanceof IntersectionType) return to.getDirectSuperTypes().allMatch(e -> isCastableTo(from, e, true));

        if (from instanceof ReferenceUnionType) return isCastableTo(from.getSuperType(), to, false);

        ReferenceType fromErased = erase(from);
        ReferenceType toErased = erase(to);
        boolean upcast;
        if ((upcast = isAssignableTo(fromErased, toErased)) || // redundant cast on erasure, type params may still prevent the cast
            isAssignableTo(toErased, fromErased)) { // to is a subclass of from

            return !provablyDistinct(
                    upcast ? from : to,
                    upcast ? to : from);
        }

        assert from instanceof ClassType;
        ClassType fromClass = (ClassType) from;
        ClassType toClass = (ClassType) to;

        if (fromClass.isInterface() && toClass.isInterface()) return true; // Always allowed.

        if (!fromClass.isInterface()) {
            if (!toClass.isInterface()) return false; // Not allowed.

            if (!fromClass.isFinal()) {
                return true; // S is not final, and T is an interface, Always allowed.
            }
            // This will never be hit, but S is a final class and must implement T.
            return isAssignableTo(fromClass, toClass);
        }
        assert fromClass.isInterface();
        if (!toClass.isFinal()) {
            return true; // T is not final, and S is an interface, Always allowed.
        }

        // T is final class, must implement S.
        return isAssignableTo(toClass, fromClass);
    }

    private static boolean provablyDistinct(ReferenceType a, ReferenceType b) {
        if (a instanceof ArrayType) {
            return b instanceof ArrayType && provablyDistinct(
                    (ReferenceType) ((ArrayType) a).getElementType(),
                    (ReferenceType) ((ArrayType) b).getElementType());
        }

        if (b instanceof ArrayType) {
            return false; // a must be Object
        }

        return provablyDistinct((ClassType) a, (ClassType) b);
    }

    /**
     * For two types where |a| <: |b|
     *
     * @return True if there is no concrete type which is assignable to both a and b
     */
    private static boolean provablyDistinct(ClassType a, ClassType b) {
        if (!(b instanceof ParameterizedClass)) return false;

        // fast bailout for |a| == |b|
        if (a.getDeclaration() == b.getDeclaration()) {
            return a instanceof ParameterizedClass && provablyDistinct((ParameterizedClass) a, (ParameterizedClass) b);
        }

        ClassType captureA = capture(a);
        ClassType bOnA = findParameterizationOrNull(b.getDeclaration(), captureA);
        if (!(bOnA instanceof ParameterizedClass)) {
            return false;
        }

        if (!provablyDistinct((ParameterizedClass) bOnA, (ParameterizedClass) b)) {
            return false;
        }

        // if a contains captures, then the parameterization of b with captured vars from a may be distinct from b
        // but there could be concrete a parameterization of a that is consistent with b and the wildcard bounds in a
        if (captureA != a && canFindConcreteParameterization((ParameterizedClass) a, (ParameterizedClass) b)) {
            return false;
        }

        return true;
    }

    private static boolean canFindConcreteParameterization(ParameterizedClass type, ParameterizedClass ancestor) {
        // of type with ancestor in the hierarchy, within the bounds of wildcads in type

        Map<TypeParameter, ReferenceType> mappings = new HashMap<>();
        ParameterizedClass p = (ParameterizedClass) makeThisType(type.getDeclaration());
        if (!mapTypes(mappings, findParameterization(ancestor.getDeclaration(), p), ancestor)) {
            return false;
        }

        // must find _concrete_ mappings, not wildcards
        if (ColUtils.anyMatch(mappings.values(), e -> e instanceof WildcardType)) {
            return false;
        }

        // check that all the mappings are within bounds
        // faster version of isAssignableTo(subst(p, mappings ?? WILDCARD), type)

        return ColUtils.allMatch(mappings.entrySet(),
                entry -> isContainedBy(entry.getValue(), type.getTypeArguments().get(entry.getKey().getIndex())));
    }

    private static boolean provablyDistinct(ParameterizedClass p1, ParameterizedClass p2) {
        List<ReferenceType> args1 = p1.getTypeArguments();
        List<ReferenceType> args2 = p2.getTypeArguments();
        for (int i = 0; i < args1.size(); i++) {
            ReferenceType arg1 = args1.get(i);
            ReferenceType arg2 = args2.get(i);
            if (!anyOverlapBetweenTypes(arg1, arg2)) {
                return true;
            }
        }

        return p1.getOuter() != null && provablyDistinct(p1.getOuter(), requireNonNull(p2.getOuter()));
    }

    private static boolean anyOverlapBetweenTypes(ReferenceType a, ReferenceType b) {
        ReferenceType upperA = properNonRecursiveUpperBound(a);
        ReferenceType upperB = properNonRecursiveUpperBound(b);
        if (!isCastableTo(upperA, upperB, true)) return false;

        // Treat type parameters as unbounded wildcards because they could be anything.
        // When used as actual types, they act instead like captures (lower bound of themselves)
        ReferenceType lowerA = a instanceof TypeParameter ? NullConstantType.INSTANCE : a.getLowerBound();
        ReferenceType lowerB = b instanceof TypeParameter ? NullConstantType.INSTANCE : b.getLowerBound();

        // ? super T extends A -> ? super A || overlap(T, ...)
        if (lowerA instanceof TypeParameter && anyOverlapBetweenTypes(lowerA, b)) return true;
        if (lowerB instanceof TypeParameter && anyOverlapBetweenTypes(a, lowerB)) return true;

        // any overlap between (upperA, lowerA) and (upperB, lowerB)
        // that is, upperB >= lowerA && upperA >= lowerB
        return isAssignableTo(lowerA, upperB, false) && isAssignableTo(lowerB, upperA, false);
    }

    private static ReferenceType properNonRecursiveUpperBound(ReferenceType a) {
        ReferenceType upper = a.getUpperBound();
        if (upper instanceof TypeVariable) return properNonRecursiveUpperBound(upper);

        if (!(a instanceof TypeVariable)) return upper;

        return TypeSubstitutions.subst(upper, t -> {
            if (!(t instanceof TypeVariable)) return t;

            return WildcardType.createExtends(t == a ? erase(upper, false) : properNonRecursiveUpperBound(t));
        });
    }
    // endregion

    // region Type operations
    public static ReferenceType erase(ReferenceType type) {
        return erase(type, true);
    }

    public static ReferenceType erase(ReferenceType type, boolean toConcreteType) {
        if (type == NullConstantType.INSTANCE) return type;

        return switch (type) {
            case IntersectionType intersection when !toConcreteType -> new IntersectionType(
                    intersection.baseType == null ? null : erase(intersection.baseType),
                    FastStream.of(intersection.getInterfaces()).map(TypeSystem::erase).toImmutableList()
            );
            case ArrayType arrayType when arrayType.getElementType() instanceof ReferenceType elemType -> arrayType.withElementType(erase(elemType, toConcreteType));
            case ArrayType arrayType -> arrayType;
            case ClassType classType -> erase(classType);
            default -> erase(type.getSuperType(), toConcreteType);
        };
    }

    public static ClassType erase(ClassType type) {
        assert isFullyDefined(type);
        return type instanceof ParameterizedClass ? type.asRaw() : type;
    }

    public static ClassType makeThisType(ClassType clazz) {
        if (!isGeneric(clazz)) return clazz;

        ParameterizedClass parameterizedOuter = null;
        if (needsOuterParameterization(clazz)) {
            parameterizedOuter = (ParameterizedClass) makeThisType(clazz.getEnclosingClass().orElseThrow(notPossible()));
        }

        return new ParameterizedClass(parameterizedOuter, clazz, unsafeCast(clazz.getTypeParameters()));
    }

    public static ReferenceType makeMultiCatchUnion(ReferenceType a, ReferenceType b) {
        if (a.equals(b)) return a;

        // flatten them!
        LinkedList<ReferenceType> types = new LinkedList<>();
        if (a instanceof ReferenceUnionType) {
            types.addAll(((ReferenceUnionType) a).getTypes());
            if (ColUtils.anyMatch(types, t -> t.equals(b))) {
                return a;
            }
        } else {
            types.add(a);
        }

        assert ColUtils.allMatch(types, t -> !isAssignableTo(b, t));
        assert !(b instanceof ReferenceUnionType);
        types.add(b);

        return new ReferenceUnionType(unsafeCast(types));
    }

    public static IntersectionType intersection(ReferenceType upperBound, ClassType interfaceType) {
        assert isInterface(interfaceType);

        if (upperBound instanceof IntersectionType intersection) {
            List<ClassType> interfaces = new ArrayList<>(intersection.getInterfaces());
            interfaces.add(interfaceType);
            return new IntersectionType(intersection.baseType, List.copyOf(interfaces));
        }

        if (isInterface(upperBound)) {
            return new IntersectionType(null, List.of((ClassType) upperBound, interfaceType));
        }

        return new IntersectionType(upperBound, List.of(interfaceType));
    }

    public static ReferenceType glb(ReferenceType a, ReferenceType b) {
        if (isObject(a)) return b;
        if (isAssignableTo(a, b)) return a;
        if (isAssignableTo(b, a)) return b;

        assert !(b instanceof IntersectionType);
        if (isInterface(b)) {
            List<ClassType> interfaces;
            ReferenceType baseType;
            if (a instanceof IntersectionType intersection) { // check if we should replace any of the interfaces with the 'more specific' interface, b
                baseType = intersection.baseType;
                interfaces = FastStream.of(intersection.getInterfaces()).filter(i -> !isAssignableTo(b, i)).toList();
            } else if (isInterface(a)) {
                baseType = null;
                interfaces = new ArrayList<>(2);
                interfaces.add((ClassType) a);
            } else {
                baseType = a;
                interfaces = new ArrayList<>(1);
            }

            interfaces.add((ClassType) b);
            return new IntersectionType(baseType, interfaces);
        }

        if (isInterface(a)) {
            return intersection(b, (ClassType) a);
        }

        if (a instanceof IntersectionType intersection) {
            assert intersection.baseType == null || isAssignableTo(b, intersection.baseType);
            return new IntersectionType(b, intersection.getInterfaces());
        }

        throw new IllegalArgumentException("Cannot find glb of " + a + " and " + b);
    }

    public static ReferenceType glb(Iterable<ReferenceType> types) {
        return FastStream.of(types)
                .fold(TypeSystem::glb)
                .orElseThrow(() -> new IllegalArgumentException("No elements supplied."));
    }

    /**
     * Ignores b if it already has a different parameterization in a
     * <p>
     * Unlike {@link #glb(ReferenceType, ReferenceType)} accepts IntersectionType for b
     * <p>
     * Not quite technically equivalent to javac, because we don't do a full supertype tree flatten,
     * but results are weird in the cases it matters
     */
    public static ReferenceType glbJavac(ReferenceType a, ReferenceType b) {
        if (b instanceof IntersectionType) {
            //noinspection ConstantConditions
            return b.getDirectSuperTypes().fold(a, TypeSystem::glbJavac);
        }

        // check assignability without erasure first, javac only flattens the hierarchies out if basic assignability fails
        if (TypeSystem.isAssignableTo(b, a)) return b;
        if (TypeSystem.isAssignableTo(a, erase(b))) return a;

        return glb(a, b);
    }

    /**
     * Ignores any types with an erasure that is a supertype of an earlier type
     * That is, if there are multiple generic parameterizations for a class [on a hierarchy], the latter will be ignored
     * <p>
     * Not quite technically equivalent to javac, because we don't do a full supertype tree flatten,
     * but results are weird in the cases it matters
     */
    public static ReferenceType glbJavac(Iterable<ReferenceType> types) {
        return FastStream.of(types)
                .fold(TypeSystem::glbJavac)
                .orElseThrow(() -> new IllegalArgumentException("No elements supplied."));
    }

    // most specific common supertype
    public static ReferenceType lub(ReferenceType... types) {
        return lub(RecursiveLCTACache.NONE, types);
    }

    interface RecursiveLCTACache {

        RecursiveLCTACache NONE = (a, b) -> null;

        @Nullable
        ReferenceType lookup(ReferenceType a, ReferenceType b);
    }

    public static ReferenceType lub(RecursiveLCTACache cache, ReferenceType... types) {
        boolean allReferenceArrays = true;
        for (ReferenceType type : types) {
            if (!(type instanceof ArrayType) || (((ArrayType) type).getElementType() instanceof PrimitiveType)) {
                allReferenceArrays = false;
                break;
            }
        }

        if (allReferenceArrays) {// LUB(arrayElemTypes)
            ReferenceType[] elemTypes = new ReferenceType[types.length];
            for (int i = 0; i < types.length; i++) {
                elemTypes[i] = (ReferenceType) ((ArrayType) types[i]).getElementType();
            }
            return ((ArrayType) types[0]).withElementType(lub(cache, elemTypes));
        }

        // we're looking for common supertypes from here on out
        LinkedList<ReferenceType> mec = getCommonDeclaredHierarchy(types);
        recoverGenericCandidates(cache, mec, types);

        return glb(mec);
    }

    private static void recoverGenericCandidates(RecursiveLCTACache cache, LinkedList<ReferenceType> mec, ReferenceType... types) {
        for (ListIterator<ReferenceType> iterator = mec.listIterator(); iterator.hasNext(); ) {
            ReferenceType c = iterator.next();
            if (!(c instanceof ClassType)) continue;
            iterator.set(recoverGenericCandidates(cache, (ClassType) c, types));
        }
    }

    private static ReferenceType recoverGenericCandidates(RecursiveLCTACache cache, ClassType decl, ReferenceType[] types) {
        if (!isGeneric(decl)) return decl;

        ClassType p = requireNonNull(findParameterization(decl, types[0]));
        for (int i = 1; i < types.length; i++) {
            p = lcp(cache, p, requireNonNull(findParameterization(decl, types[i])));
        }

        return p;
    }

    private static ClassType lcp(RecursiveLCTACache cache, ClassType c1, ClassType c2) {
        if (c1.equals(c2)) return c1;
        if (c1 instanceof RawClass) return c1;
        if (c2 instanceof RawClass) return c2;
        return lcp(cache, (ParameterizedClass) c1, (ParameterizedClass) c2);
    }

    private static ParameterizedClass lcp(RecursiveLCTACache cache, ParameterizedClass c1, ParameterizedClass c2) {
        assert c1.getDeclaration() == c2.getDeclaration();
        // tedious because of inner classes
        List<ReferenceType> args1 = c1.getTypeArguments();
        List<ReferenceType> args2 = c2.getTypeArguments();
        List<ReferenceType> args = new ArrayList<>(args1.size());
        for (int i = 0; i < args1.size(); i++) {
            ReferenceType t1 = args1.get(i);
            ReferenceType t2 = args2.get(i);

            args.add(lcta(cache, t1, t2));
        }

        ParameterizedClass outer = c1.getOuter() == null ? null : lcp(cache, c1.getOuter(), requireNonNull(c2.getOuter()));
        return new ParameterizedClass(outer, c1.getDeclaration(), args);
    }

    private static ReferenceType lcta(RecursiveLCTACache cache, ReferenceType t1, ReferenceType t2) {
        ReferenceType t1Bound = t1 instanceof CapturedTypeVar ? ((CapturedTypeVar) t1).wildcard : t1;
        ReferenceType t2Bound = t2 instanceof CapturedTypeVar ? ((CapturedTypeVar) t2).wildcard : t2;
        if (isContainedBy(t1Bound, t2Bound)) return t2Bound;
        if (isContainedBy(t2Bound, t1Bound)) return t1Bound;

        if (t1 instanceof WildcardType) return lcta(cache, t1.getUpperBound(), t2);
        if (t2 instanceof WildcardType) return lcta(cache, t1, t2.getUpperBound());

        ReferenceType cached = cache.lookup(t1, t2);
        // proper implementation of jls, infinitely recursive type:
        //if (cached != null) return cached;
        // javac version:
        if (cached != null) return WildcardType.createExtends(objectType(t1));

        class InfiniteWildcard extends WildcardType {

            InfiniteWildcard() {
                super(NullConstantType.INSTANCE, NullConstantType.INSTANCE);
            }

            @Override
            public boolean isInfinite() {
                return true;
            }

            @Override
            public String getFullName() {
                return "? extends lub(" + t1 + "," + t2 + ")";
            }
        }

        InfiniteWildcard result = new InfiniteWildcard();
        result.upperBound = lub(
                (a, b) -> a.equals(t1) && b.equals(t2) ? result : cache.lookup(a, b),
                t1,
                t2
        );

        return WildcardType.createExtends(result.upperBound);
    }

    private static void intersect(LinkedList<ReferenceType> a, List<ReferenceType> b) {
        a.removeIf(t -> !ColUtils.anyMatch(b, t::equals));
    }

    private static LinkedList<ReferenceType> minimal(LinkedList<ReferenceType> types) {
        LinkedList<ReferenceType> minimal = new LinkedList<>();
        outer:
        while (!types.isEmpty()) {
            ReferenceType t = types.removeFirst();
            for (ReferenceType t2 : types) {
                if (isSubType(t2, t)) {
                    continue outer;
                }
            }
            minimal.addLast(t);
        }
        return minimal;
    }

    public static LinkedList<ReferenceType> getCommonDeclaredHierarchy(ReferenceType... types) {
        LinkedList<ReferenceType> h = getDeclaredHierarchy(types[0]);
        for (int i = 1; i < types.length; i++) {
            intersect(h, getDeclaredHierarchy(types[i]));
        }
        return minimal(h);
    }

    private static LinkedList<ReferenceType> getDeclaredHierarchy(ReferenceType type) {
        LinkedList<ReferenceType> l = new LinkedList<>();
        getDeclaredHierarchy(l, type);
        return l;
    }

    private static void getDeclaredHierarchy(LinkedList<ReferenceType> l, ReferenceType type) {
        if (type instanceof ClassType) {
            type = ((ClassType) type).getDeclaration();
        }
        if (l.contains(type)) return;

        for (ReferenceType superType : type.getDirectSuperTypes()) {
            getDeclaredHierarchy(l, superType);
        }

        l.add(type);
    }

    public static boolean mapTypes(Map<TypeParameter, ReferenceType> mappings, ReferenceType t1, ReferenceType t2) {
        if (t1 instanceof TypeParameter) {
            assert !mappings.containsKey(t1) || mappings.get(t1).equals(t2);
            mappings.put((TypeParameter) t1, t2);
            return true;
        }

        if (t1 instanceof ArrayType) {
            if (!(t2 instanceof ArrayType)) return false;

            return mapTypes(mappings, (ReferenceType) ((ArrayType) t1).getElementType(), (ReferenceType) ((ArrayType) t2).getElementType());
        }

        if (t1 instanceof ParameterizedClass) {
            if (!(t2 instanceof ParameterizedClass)) return false;

            List<ReferenceType> p1 = ((ParameterizedClass) t1).getTypeArguments();
            List<ReferenceType> p2 = ((ParameterizedClass) t2).getTypeArguments();
            boolean ret = true;
            for (int i = 0; i < p1.size(); i++) {
                ret &= mapTypes(mappings, p1.get(i), p2.get(i));
            }
            return ret;
        }

        return t1.equals(t2);
    }

    public static ClassType box(TypeResolver resolver, PrimitiveType type) {
        return resolver.resolveClass(type.getBoxedClass());
    }

    @Nullable
    public static PrimitiveType unbox(ReferenceType type) {
        return UNBOX_LOOKUP.get(type.getFullName());
    }
    // endregion

    // region Type Queries
    public static boolean isFullyDefined(ReferenceType type) {
        if (type instanceof WildcardType) return false;
        if (type instanceof ClassType) return isFullyDefined((ClassType) type);
        return true;
    }

    public static boolean isFullyDefined(ClassType type) {
        if (type instanceof ParameterizedClass) {
            return ((ParameterizedClass) type).isFullyParameterized();
        }

        return !isGeneric(type);
    }

    public static boolean isFullyDefined(Method method) {
        if (method instanceof ParameterizedMethod) {
            return ((ParameterizedMethod) method).isFullyParameterized();
        }

        return !method.hasTypeParameters();
    }

    public static boolean isConstructedViaTargetInstance(ClassType clazz) {
        if (clazz.getDeclType() == ClassType.DeclType.ANONYMOUS) { return isConstructedViaTargetInstance(clazz.getSuperClass()); }

        if (clazz.getDeclType() != ClassType.DeclType.INNER) return false;
        if (clazz.isStatic()) return false;
        if (clazz.getEnclosingClass().isEmpty()) return false; // you can have non-static inner classes which are 'helper' classes and don't have an outer (as per the inner class file)
        return true;
    }

    public static boolean needsOuterParameterization(ClassType clazz) {
        return clazz.getDeclType() == ClassType.DeclType.INNER
               && !clazz.isStatic()
               && isGeneric(clazz.getEnclosingClass().orElseThrow(notPossible()));
    }

    public static boolean isGeneric(ClassType clazz) {
        return clazz.hasTypeParameters() || needsOuterParameterization(clazz);
    }

    public static boolean areErasuresEqual(AType a, AType b) {
        // TODO, could be more efficient, don't actually need to _do_ the erasure, just check the type decls
        return a.equals(b) || // fast bailout
               a instanceof ReferenceType && b instanceof ReferenceType && erase((ReferenceType) a).equals(erase((ReferenceType) b));
    }

    public static boolean isInterface(ReferenceType t) {
        return t instanceof ClassType && ((ClassType) t).isInterface();
    }

    public static boolean isIntegerConstant(AType type) {
        return type instanceof IntegerConstantType || type instanceof IntegerConstantUnion;
    }

    public static ClassType findParameterization(ClassType decl, ReferenceType inType) {
        return requireNonNull(findParameterizationOrNull(decl, inType));
    }

    @Nullable
    public static ClassType findParameterizationOrNull(ClassType decl, ReferenceType inType) {
        if (getDecl(inType) == decl) return (ClassType) inType;
        return inType.getDirectSuperTypes().map(e -> findParameterizationOrNull(decl, e)).filter(Objects::nonNull).firstOrDefault();
    }

    // region Java lang types
    public static boolean isObject(AType type) {
        return type instanceof ClassType && type.getFullName().equals("java.lang.Object");
    }

    public static boolean isString(AType type) {
        return type instanceof ClassType && type.getFullName().equals("java.lang.String");
    }
    // endregion
    // endregion

    // region Helpers
    public static ClassType objectType(ReferenceType type) {
        return isObject(type) ? (ClassType) type : objectType(type.getSuperType());
    }

    public static TypeResolver resolver(ReferenceType type) {
        return objectType(type).getTypeResolver();
    }

    public static AType capture(AType type) {
        return type instanceof ReferenceType ? capture((ReferenceType) type) : type;
    }

    public static ReferenceType capture(ReferenceType type) {
        return type instanceof ParameterizedClass ? capture((ParameterizedClass) type) : type;
    }

    public static ClassType capture(ClassType type) {
        return type instanceof ParameterizedClass ? capture((ParameterizedClass) type) : type;
    }

    public static ParameterizedClass capture(ParameterizedClass type) {
        return type.capture();
    }

    private static ReferenceType getDecl(ReferenceType e) {
        return e instanceof ClassType ? ((ClassType) e).getDeclaration() : e;
    }
    // endregion
}
