package net.covers1624.coffeegrinder.util.resolver;

import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.net.URI;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.Map;

/**
 * Uses the built-in dynamically loaded Java Run Time FS.
 * <p>
 * This FS loads itself _from_ the target Java Home, loading
 * the JImage modules file or exploded directories.
 * <p>
 * Created by covers1624 on 11/20/25.
 */
public class JRTResolver implements Resolver {

    private final FileSystem fs;
    private final List<JRTModule> modules;

    public JRTResolver(Path javaHome) throws IOException {
        fs = FileSystems.newFileSystem(
                URI.create("jrt:///"),
                Map.of("java.home", javaHome.toAbsolutePath().normalize().toString())
        );

        try (var stream = Files.list(fs.getPath("/modules/"))) {
            modules = FastStream.of(stream)
                    .map(JRTModule::new)
                    .toImmutableList();
        }
    }

    @Override
    public FastStream<String> getAllClasses() {
        return FastStream.of(modules)
                .flatMap(Resolver::getAllClasses);
    }

    @Override
    public FastStream<String> getAllResources() {
        return FastStream.of(modules)
                .flatMap(Resolver::getAllResources);
    }

    @Override
    public byte @Nullable [] getResource(String name) throws IOException {
        for (var module : modules) {
            var bytes = module.getResource(name);
            if (bytes != null) {
                return bytes;
            }
        }
        return null;
    }

    @Override
    public boolean hasResource(String name) {
        for (var module : modules) {
            if (module.hasResource(name)) {
                return true;
            }
        }
        return false;
    }

    @Override
    public void close() throws IOException {
        fs.close();
    }

    private static final class JRTModule extends FolderResolver {

        JRTModule(Path path) {
            super(path);
        }

        @Override
        public String toString() {
            return "JRTModule(" + path + ")";
        }
    }
}
