package net.covers1624.coffeegrinder.bytecode.insns;

import net.covers1624.coffeegrinder.bytecode.InsnVisitor;
import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.InstructionFlag;
import net.covers1624.coffeegrinder.type.AType;
import net.covers1624.coffeegrinder.type.AnnotationSupplier;
import net.covers1624.coffeegrinder.util.EnumBitSet;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

/**
 * Represents a variable within a Method.
 * <p>
 * Created by covers1624 on 23/2/21.
 */
public class LocalVariable extends Instruction {

    /**
     * The {@link VariableKind} this variable represents.
     */
    private final VariableKind kind;

    private @Nullable String genericSignature;

    /**
     * The {@link AType} of this variable.
     */
    private AType type;

    /**
     * The index of this variable.
     * <p>
     * This will only be set for {@link VariableKind#LOCAL Locals} and {@link VariableKind#PARAMETER Parameters}.
     */
    private final int index;

    /**
     * This variables name.
     */
    private String name;

    /**
     * An id for differentiating multiple variables with the same name within the AST.
     */
    private int subId;

    private final List<LocalReference> references = new LinkedList<>();

    /**
     * If this variable does not have an associated LocalVariable table entry
     * and may represent a purely synthetic compiler-generated variable.
     * <p>
     * This will only be set for {@link VariableKind#LOCAL Locals}.
     */
    private boolean isSynthetic;

    private AnnotationSupplier annotationSupplier = AnnotationSupplier.EMPTY;

    public LocalVariable(VariableKind kind, AType type, String name) {
        this(kind, type, null, -1, name);
    }

    public LocalVariable(VariableKind kind, AType type, @Nullable String genericSignature, int index, String name) {
        this.kind = kind;
        this.type = type;
        this.genericSignature = genericSignature;
        this.index = index;
        this.name = name;
        isSynthetic = kind == VariableKind.STACK_SLOT;
        assert kind == VariableKind.LOCAL || kind == VariableKind.PARAMETER ? index >= 0 : index == -1;
        assert kind != VariableKind.PARAMETER || this instanceof ParameterVariable;
    }

    public void addReference(LocalReference insn) {
        references.add(insn);
        if (!isConnected()) {
            throw new UnsupportedOperationException("Cannot add reference to a disconnected variable.");
        }
    }

    public void removeReference(LocalReference insn) {
        references.remove(insn);
        if (isDead() && isConnected()) {
            remove();
        }
    }

    @Override
    public EnumBitSet<InstructionFlag> getDirectFlags() {
        return InstructionFlag.NONE;
    }

    @Override
    public <R, C> R accept(InsnVisitor<R, C> visitor, C ctx) {
        return visitor.visitLocalVariable(this, ctx);
    }

    @Override
    public AType getResultType() {
        throw new UnsupportedOperationException();
    }

    //@formatter:off
    public VariableKind getKind() { return kind; }
    @Nullable public String getGenericSignature() { return genericSignature; }
    public AType getType() { return type; }
    public int getIndex() { return index; }
    public String getName() { return name; }
    public int getSubId() { return subId; }
    public List<LocalReference> getReferences() { return Collections.unmodifiableList(references); }
    public boolean isSynthetic() { return isSynthetic; }
    public AnnotationSupplier getAnnotationSupplier() { return annotationSupplier; }
    public void setGenericSignature(@Nullable String genericSignature) { this.genericSignature = genericSignature; }
    public void setType(AType type) { this.type = type; }
    public void setName(String name) { this.name = name; }
    public void setSubId(int subId) { this.subId = subId; }
    public void setSynthetic(boolean synthetic) { isSynthetic = synthetic; }
    public void setAnnotationSupplier(AnnotationSupplier annotationSupplier) { this.annotationSupplier = annotationSupplier; }
    //@formatter:on

    /**
     * The number of instructions currently referencing this variable which read from it.
     *
     * @return The number of read references.
     */
    public int getLoadCount() {
        return FastStream.of(references)
                .filter(Reference::isReadFrom)
                .count();
    }

    /**
     * The number of instructions currently referencing this variable which write to it.
     *
     * @return The number of write references.
     */
    public int getStoreCount() {
        return FastStream.of(references)
                .filter(Reference::isWrittenTo)
                .count();
    }

    /**
     * Gets the total number of references this variable has.
     *
     * @return The number of references.
     */
    public int getReferenceCount() {
        return references.size();
    }

    /**
     * If this variable can be considered dead.
     * <p>
     * A variable is considered dead if:
     * 1, It is not a parameter.
     * 2, The variable has no stores.
     *
     * @return If the variable is dead.
     */
    public boolean isDead() {
        // Note, an invariant makes sure no variables can exist with only load/declare references
        return kind != VariableKind.PARAMETER && references.isEmpty();
    }

    public String getUniqueName() {
        if (getSubId() > 0) {
            return getName() + "$" + getSubId();
        }
        return getName();
    }

    public enum VariableKind {
        LOCAL,
        PARAMETER,
        STACK_SLOT,
        TEMP_LOCAL,
    }

}
