package net.covers1624.coffeegrinder.bytecode.matching;

import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.type.AType;
import net.covers1624.coffeegrinder.type.Field;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.Nullable;

import java.util.Objects;

/**
 * Created by covers1624 on 19/4/21.
 */
public class LoadStoreMatching {

    /**
     * Matches the given instruction to a {@link Store} instruction, whose child is a
     * {@link Load} instruction of a specific variable.
     *
     * @param insn     The insn to match against.
     * @param variable The variable to match against.
     * @return The {@link Store} instruction or <code>null</code>
     */
    @Nullable
    public static Store matchStoreLocalLoadLocal(@Nullable Instruction insn, LocalVariable variable) {
        Store store = matchStoreLocal(insn);
        if (store == null) return null;

        Load load = matchLoadLocal(store.getValue(), variable);
        if (load == null) return null;

        return store;
    }

    @Nullable
    public static LocalReference matchLocalRef(@Nullable Instruction insn, LocalVariable variable) {
        if (!(insn instanceof LocalReference localRef) || localRef.variable != variable) return null;
        return localRef;
    }

    /**
     * Matches the given instruction to a Load instruction.
     *
     * @param insn The insn to match against.
     * @return The Load instruction.
     */
    @Nullable
    @Contract ("null->null")
    public static Load matchLoadLocal(@Nullable Instruction insn) {
        if (!(insn instanceof Load load) || !(load.getReference() instanceof LocalReference)) return null;
        return load;
    }

    /**
     * Matches the given instruction to a Load instruction of a specific variable.
     *
     * @param insn     The insn to match against.
     * @param variable The variable to match against.
     * @return The Load instruction.
     */
    @Nullable
    public static Load matchLoadLocal(@Nullable Instruction insn, LocalVariable variable) {
        Load load = matchLoadLocal(insn);
        if (load == null) return null;

        if (load.getVariable() != variable) return null;
        return load;
    }

    /**
     * Matches the given instruction to a Store instruction.
     *
     * @param insn The insn to match against.
     * @return The Store instruction.
     */
    @Nullable
    public static Store matchStoreLocal(@Nullable Instruction insn) {
        if (!(insn instanceof Store store) || !(store.getReference() instanceof LocalReference)) return null;
        return store;
    }

    /**
     * Matches the given instruction to a Store instruction of a specific variable.
     *
     * @param insn     The insn to match against.
     * @param variable The variable to match against.
     * @return The Store instruction.
     */
    @Nullable
    public static Store matchStoreLocal(@Nullable Instruction insn, LocalVariable variable) {
        Store store = matchStoreLocal(insn);
        if (store == null) return null;

        if (store.getVariable() != variable) return null;
        return store;
    }

    @Nullable
    public static FieldReference matchFieldRef(@Nullable Instruction insn, Field field) {
        ;
        if (!(insn instanceof FieldReference fieldRef)) return null;
        if (!matchField(fieldRef.getField(), field)) return null;

        return fieldRef;
    }

    /**
     * Matches the given instruction to a LoadField of the given field.
     *
     * @param insn The instruction to match.
     * @return The LoadField or <code>null</code>.
     */
    @Nullable
    public static Load matchLoadField(@Nullable Instruction insn) {
        if (!(insn instanceof Load load)) return null;
        if (!(load.getReference() instanceof FieldReference)) return null;

        return load;
    }

    /**
     * Matches the given instruction to a LoadField of the given field.
     *
     * @param insn  The instruction to match.
     * @param field The field to match.
     * @return The LoadField or <code>null</code>.
     */
    @Nullable
    public static Load matchLoadField(@Nullable Instruction insn, Field field) {
        if (!(insn instanceof Load load)) return null;
        if (matchFieldRef(load.getReference(), field) == null) return null;

        return load;
    }

    @Nullable
    public static FieldReference matchLoadFieldRef(@Nullable Instruction insn) {
        if (!(insn instanceof Load load)) return null;
        if (!(load.getReference() instanceof FieldReference fieldRef)) return null;
        return fieldRef;
    }

    @Nullable
    public static FieldReference matchStoreFieldRef(@Nullable Instruction insn) {
        if (!(insn instanceof Store store)) return null;
        if (!(store.getReference() instanceof FieldReference fieldRef)) return null;
        return fieldRef;
    }

    /**
     * Matches the given instruction to a StoreField of the given field.
     *
     * @param insn The instruction to match.
     * @return The StoreField or <code>null</code>.
     */
    @Nullable
    public static Store matchStoreField(@Nullable Instruction insn) {
        if (!(insn instanceof Store store) || !(store.getReference() instanceof FieldReference)) return null;

        return store;
    }

    /**
     * Matches the given instruction to a StoreField of the given field.
     *
     * @param insn  The instruction to match.
     * @param field The field to match.
     * @return The StoreField or <code>null</code>.
     */
    @Nullable
    @Contract ("null,_->null")
    public static Store matchStoreField(@Nullable Instruction insn, Field field) {
        if (!(insn instanceof Store store)) return null;
        if (matchFieldRef(store.getReference(), field) == null) return null;
        return store;
    }

    public static boolean matchField(Field field1, Field field2) {
        return field1.getName().equals(field2.getName())
               && field1.getDescriptor().equals(field2.getDescriptor())
               // short-circuit a type system resolve if the references are identical
               && (field1.getDeclaringClass().equals(field2.getDeclaringClass()) || field1.equals(field2));
    }

    public static boolean equivalentFieldDescriptors(Field field1, Field field2) {
        return field1.getName().equals(field2.getName())
               && field1.getDescriptor().equals(field2.getDescriptor())
               // short-circuit a type system resolve if the references are identical
               && field1.getDeclaringClass().equals(field2.getDeclaringClass());
    }

    /**
     * Matches the given instruction to an {@link ArrayLen} instruction which
     * contains a {@link Load} of the given variable.
     *
     * @param insn    The insn to match against.
     * @param loadVar The variable loaded by the {@link ArrayLen} target.
     * @return The {@link ArrayLen} instruction.
     */
    @Nullable
    public static ArrayLen matchArrayLenLoad(@Nullable Instruction insn, LocalVariable loadVar) {
        if (!(insn instanceof ArrayLen arrayLen)) return null;
        Load load = matchLoadLocal(arrayLen.getArray(), loadVar);
        if (load == null) return null;

        return arrayLen;
    }

    @Nullable
    public static ArrayElementReference matchLoadElemRef(@Nullable Instruction insn) {
        if (!(insn instanceof Load load)) return null;
        if (!(load.getReference() instanceof ArrayElementReference arrayRef)) return null;

        return arrayRef;
    }

    @Nullable
    public static Load matchPop(@Nullable Instruction insn) {
        if (!(matchLoadLocal(insn) instanceof Load load)) return null;
        if (load.getVariable().getKind() != LocalVariable.VariableKind.STACK_SLOT) return null;
        return load;
    }

    @Nullable
    public static Load matchPop(@Nullable Instruction insn, LocalVariable variable) {
        if (!(matchPop(insn) instanceof Load load)) return null;
        if (load.getVariable() != variable) return null;
        return load;
    }

    @Nullable
    public static Store matchPush(@Nullable Instruction insn) {
        if (!(matchStoreLocal(insn) instanceof Store store)) return null;
        if (store.getVariable().getKind() != LocalVariable.VariableKind.STACK_SLOT) return null;
        return store;
    }

    @Nullable
    public static Store matchPush(@Nullable Instruction insn, LocalVariable variable) {
        if (!(matchPush(insn) instanceof Store store)) return null;
        if (store.getVariable() != variable) return null;
        return store;
    }

    /**
     * Given a stack pop (Load), will attempt find the associated stack push (Store).
     * <p>
     * If multiple stack pushes (Stores) exist, will return {@code null}.
     *
     * @param pop The stack pop.
     * @return The value which was pushed.
     */
    public static @Nullable Instruction matchPushForPop(@Nullable Instruction pop) {
        if (!(matchPop(pop) instanceof Load load)) return null;
        var variable = load.getVariable();

        // We want to find a single store.
        if (variable.getStoreCount() != 1) return null;

        var store = FastStream.of(variable.getReferences())
                .map(e -> matchPush(e.getParent()))
                .filter(Objects::nonNull)
                .onlyOrDefault();
        if (store == null) return null;

        return store.getValue();
    }

    public static @Nullable Instruction matchPushForPop(@Nullable Instruction push, @Nullable Instruction pop) {
        if (!(matchPop(pop) instanceof Load load)) return null;
        if (!(matchPush(push, load.getVariable()) instanceof Store store)) return null;

        return store.getValue();
    }

    public static @Nullable Cast matchStoreCast(@Nullable Instruction insn, @Nullable AType type) {
        if (!(matchStoreLocal(insn) instanceof Store store)) return null;
        if (!(store.getValue() instanceof Cast cast)) return null;
        if (!cast.getType().equals(type)) return null;

        return cast;
    }
}
