package net.covers1624.coffeegrinder.bytecode.transform.transformers.statement;

import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.transform.StatementTransformContext;
import net.covers1624.coffeegrinder.bytecode.transform.StatementTransformer;
import net.covers1624.coffeegrinder.bytecode.transform.transformers.PrepareSwitchOnType;
import net.covers1624.coffeegrinder.type.ClassType;
import net.covers1624.coffeegrinder.type.TypeResolver;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;

import java.util.Objects;

import static java.util.Objects.requireNonNull;
import static net.covers1624.coffeegrinder.bytecode.matching.LdcMatching.matchLdcInt;
import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.matchLoadLocal;
import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.matchStoreLocal;

/**
 * Created by covers1624 on 1/22/26.
 */
public class FinishSwitchOnType implements StatementTransformer {

    private final @Nullable ClassType switchBootstraps;

    public FinishSwitchOnType(TypeResolver resolver) {
        switchBootstraps = resolver.tryResolveClassDecl("java/lang/runtime/SwitchBootstraps");
    }

    @Override
    public void transform(Instruction statement, StatementTransformContext ctx) {
        // Only process if we have SwitchBootstraps
        if (switchBootstraps == null) return;

        // Match the following pattern:
        // Object selVar = ...
        // int indexVar = 0
        // [... expression] switch (INVOKEDYNAMIC SwitchBootstraps.typeSwitch(selVar, indexVar)) { }
        if (!(matchStoreLocal(statement) instanceof Store selVarStore)) return;
        var pattVar = selVarStore.getVariable();

        if (!(matchStoreLocal(selVarStore.getNextSiblingOrNull()) instanceof Store indexVarStore)) return;
        var indexVar = indexVarStore.getVariable();
        if (matchLdcInt(indexVarStore.getValue(), 0) == null) return;

        var swtch = matchPatternSwitchDecl(indexVarStore.getNextSiblingOrNull(), pattVar, indexVar);
        if (swtch == null) return;

        assert indexVar.getLoadCount() == 1;
        assert pattVar.getLoadCount() == 1;

        ctx.pushStep("Cleanup switch-on-type " + swtch.getBody().getEntryPoint().getName());
        // Nuke the indy and synthetic variables.
        swtch.getValue().replaceWith(selVarStore.getValue());
        selVarStore.remove();
        indexVarStore.remove();
        ctx.popStep();
    }

    private @Nullable Switch matchPatternSwitchDecl(@Nullable Instruction insn, LocalVariable pattVar, LocalVariable indexVar) {
        if (insn instanceof Switch swtch) {
            if (matchTypeSwitchIndy(swtch.getValue(), pattVar, indexVar) == null) return null;
            return swtch;
        }

        // The switch may have already been turned into a switch expression and inlined somewhere, lets go hunting!
        Instruction indyCandidate = FastStream.of(pattVar.getReferences())
                .filter(e -> e.isDescendantOf(insn))
                .map(e -> matchLoadLocal(e.getParent(), pattVar))
                .filter(Objects::nonNull)
                .map(e -> matchTypeSwitchIndy(e.getParent(), pattVar, indexVar))
                .filter(Objects::nonNull)
                .onlyOrDefault();
        if (indyCandidate == null) return null;

        return indyCandidate.getParent() instanceof Switch swtch ? swtch : null;
    }

    private @Nullable InvokeDynamic matchTypeSwitchIndy(Instruction insn, LocalVariable pattVar, LocalVariable indexVar) {
        var indy = PrepareSwitchOnType.matchTypeSwitchIndy(insn, requireNonNull(switchBootstraps));
        if (indy == null) return null;

        if (matchLoadLocal(indy.arguments.get(0), pattVar) == null) return null;
        if (matchLoadLocal(indy.arguments.get(1), indexVar) == null) return null;
        return indy;
    }
}
