package net.covers1624.coffeegrinder.bytecode.insns;

import net.covers1624.coffeegrinder.bytecode.*;
import net.covers1624.coffeegrinder.bytecode.transform.transformers.IntegerConstantInference;
import net.covers1624.coffeegrinder.type.AType;
import net.covers1624.coffeegrinder.type.PrimitiveType;
import net.covers1624.coffeegrinder.type.TypeSystem;
import net.covers1624.coffeegrinder.util.EnumBitSet;

/**
 * Created by covers1624 on 2/3/21.
 */
public class Binary extends Instruction {

    private BinaryOp op;
    private final InstructionSlot<Instruction> left = new InstructionSlot<>(this);
    private final InstructionSlot<Instruction> right = new InstructionSlot<>(this);

    public Binary(BinaryOp op, Instruction left, Instruction right) {
        super(InsnOpcode.BINARY);
        this.op = op;
        this.left.set(left);
        this.right.set(right);
    }

    @Override
    public EnumBitSet<InstructionFlag> getDirectFlags() {
        AType lh = left.get().getResultType();
        // Integers are special and numeric operations throw exceptions in some cases.
        boolean doesThrow = (lh == PrimitiveType.INT || lh == PrimitiveType.LONG) && (op == BinaryOp.DIV || op == BinaryOp.REM);
        return doesThrow ? InstructionFlag.MAY_THROW.toSet() : InstructionFlag.NONE;
    }

    @Override
    public AType getResultType() {
        AType leftType = left.get().getResultType();
        AType rightType = right.get().getResultType();

        // String concat can be between a string and a primitive.
        if (TypeSystem.isString(leftType)) return leftType;
        if (TypeSystem.isString(rightType)) return rightType;

        if (leftType.equals(PrimitiveType.BOOLEAN) || rightType.equals(PrimitiveType.BOOLEAN)) return PrimitiveType.BOOLEAN;
        if (op.isLogic() && IntegerConstantInference.isBooleanConstant(leftType) && IntegerConstantInference.isBooleanConstant(rightType)) {
            // a more correct implementation would actually compute the combinations of ops on left and right
            // but for our purposes, it's okay to return IntConstantUnion(0 | 1) even if it could only be 0, or only be 1
            return IntegerConstantInference.BOOLEAN_CONSTANTS;
        }

        if (TypeSystem.isAssignableTo(leftType, PrimitiveType.INT) && TypeSystem.isAssignableTo(rightType, PrimitiveType.INT)) return PrimitiveType.INT;
        if (TypeSystem.isAssignableTo(leftType, PrimitiveType.LONG) && TypeSystem.isAssignableTo(rightType, PrimitiveType.LONG)) return PrimitiveType.LONG;
        if (TypeSystem.isAssignableTo(leftType, PrimitiveType.FLOAT) && TypeSystem.isAssignableTo(rightType, PrimitiveType.FLOAT)) return PrimitiveType.FLOAT;
        if (TypeSystem.isAssignableTo(leftType, PrimitiveType.DOUBLE) && TypeSystem.isAssignableTo(rightType, PrimitiveType.DOUBLE)) return PrimitiveType.DOUBLE;

        throw new IllegalStateException("Conflicting types for NumericInsn." + op + " (" + leftType + ", " + rightType + ")");
    }

    @Override
    public <R, C> R accept(InsnVisitor<R, C> visitor, C ctx) {
        return visitor.visitBinary(this, ctx);
    }

    //@formatter:off
    public BinaryOp getOp() { return op; }
    public Instruction getLeft() { return left.get(); }
    public Instruction getRight() { return right.get(); }
    public void setOp(BinaryOp op) { this.op = op;invalidateFlags(); }
    public void setLeft(Instruction left) { this.left.set(left); }
    public void setRight(Instruction right) { this.right.set(right); }
    //@formatter:on
}
