package net.covers1624.coffeegrinder.bytecode.matching;

import net.covers1624.coffeegrinder.bytecode.InsnOpcode;
import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import org.jetbrains.annotations.Nullable;

import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.matchLoadLocal;

/**
 * Created by covers1624 on 19/4/21.
 */
public class BranchLeaveMatching {

    /**
     * Attempts to match a {@link Return} instruction that is exiting a function.
     *
     * @param insn The instruction.
     * @return The {@link Return} instruction, otherwise <code>null</code>.
     */
    @Nullable
    public static Return matchReturn(@Nullable Instruction insn) {
        if (insn == null || insn.opcode != InsnOpcode.RETURN) return null;
        return (Return) insn;
    }

    @Nullable
    public static Return matchReturn(@Nullable Instruction insn, MethodDecl method) {
        if (insn == null || insn.opcode != InsnOpcode.RETURN) return null;
        Return ret = (Return) insn;
        return ret.getMethod() == method ? ret : null;
    }

    /**
     * Attempts to match a leave instruction.
     *
     * @param insn The instruction.
     * @return The Leave instruction, otherwise empty.
     */
    @Nullable
    public static Leave matchLeave(@Nullable Instruction insn) {
        if (insn == null || insn.opcode != InsnOpcode.LEAVE) return null;
        return (Leave) insn;
    }

    /**
     * Attempts to match a leave instruction which leaves the specified BlockContainer.
     *
     * @param insn      The instruction.
     * @param container The BlockContainer.
     * @return The Leave instruction, otherwise empty.
     */
    @Nullable
    public static Leave matchLeave(@Nullable Instruction insn, BlockContainer container) {
        Leave leave = matchLeave(insn);
        if (leave == null) return null;

        if (leave.getTargetContainer() != container) return null;
        return leave;
    }

    /**
     * Attempts to match a branch instruction.
     *
     * @param insn The instruction.
     * @return The Branch instruction, otherwise empty.
     */
    @Nullable
    public static Branch matchBranch(@Nullable Instruction insn) {
        if (!(insn instanceof Branch)) return null;
        return (Branch) insn;
    }

    /**
     * Attempts to match a branch instruction to a given block.
     *
     * @param insn        The instruction.
     * @param targetBlock The target block.
     * @return The Branch instruction, otherwise empty.
     */
    @Nullable
    public static Branch matchBranch(@Nullable Instruction insn, Block targetBlock) {
        Branch branch = matchBranch(insn);
        if (branch == null) return null;
        if (branch.getTargetBlock() != targetBlock) return null;
        return branch;
    }

    /**
     * Returns true if exit1 and exit2 are both exit instructions
     * (branch or leave) and both represent the same exit.
     *
     * @param exit1 The first exit point.
     * @param exit2 The second exit point.
     * @return If they represent the same exit.
     */
    public static boolean compatibleExitInstruction(@Nullable Instruction exit1, @Nullable Instruction exit2) {
        if (exit1 == null || exit2 == null || exit1.opcode != exit2.opcode) return false;

        switch (exit1.opcode) {
            case BRANCH: {
                Branch br1 = (Branch) exit1;
                Branch br2 = (Branch) exit2;
                return br1.getTargetBlock() == br2.getTargetBlock();
            }
            case LEAVE: {
                Leave l1 = (Leave) exit1;
                Leave l2 = (Leave) exit2;
                return l1.getTargetContainer() == l2.getTargetContainer();
            }
            default:
                return false;
        }
    }

    /**
     * Attempts to match a {@link Throw} instruction.
     *
     * @param insn The Instruction.
     * @return The {@link Throw} or <code>null</code>.
     */
    @Nullable
    public static Throw matchThrow(@Nullable Instruction insn) {
        if (insn == null || insn.opcode != InsnOpcode.THROW) return null;

        return (Throw) insn;
    }

    /**
     * Attempts to match a {@link Throw} instruction, which loads
     * the specified variable.
     *
     * @param insn     The Instruction.
     * @param variable The variable.
     * @return The {@link Throw} or <code>null</code>.
     */
    @Nullable
    public static Throw matchThrow(@Nullable Instruction insn, LocalVariable variable) {
        Throw thr = matchThrow(insn);
        if (thr == null) return null;

        if (matchLoadLocal(thr.getArgument(), variable) == null) return null;

        return thr;
    }
}
