package net.covers1624.coffeegrinder.type;

import java.util.function.Function;

import static net.covers1624.quack.collection.FastStream.of;

/**
 * Helpers for applying type substitutions.
 * <p>
 * Created by covers1624 on 2/5/22.
 */
public class TypeSubstitutions {

    /**
     * Apply the substitutions given by the provided mapper recursively.
     *
     * @param type   The input type.
     * @param mapper The {@link TypeMapper}.
     * @return The substituted type.
     */
    public static AType subst(AType type, TypeMapper mapper) {
        if (type instanceof ReferenceType) {
            return subst((ReferenceType) type, mapper);
        }
        return type;
    }

    /**
     * Apply the substitutions given by the provided mapper recursively.
     *
     * @param type   The input type.
     * @param mapper The {@link TypeMapper}.
     * @return The substituted type.
     */
    public static ReferenceType subst(ReferenceType type, TypeMapper mapper) {
        if (type == NullConstantType.INSTANCE) return type;

        boolean wasTypeParam = type instanceof TypeParameter;
        type = mapper.mapType(type);
        if (wasTypeParam) {
            return type; // temporary bailout for recursive types, ideally we should detect recursion inside here.
        }

        if (type instanceof CapturedTypeVar) {
            return ((CapturedTypeVar) type).map(mapper);
        }

        if (type instanceof TypeVariable) {
            return type;
        }

        if (type instanceof ArrayType) {
            ArrayType arrayType = (ArrayType) type;
            if (!(arrayType.getElementType() instanceof ReferenceType)) return arrayType;

            ReferenceType elemType = subst((ReferenceType) arrayType.getElementType(), mapper);
            return arrayType.withElementType(elemType instanceof  WildcardType ? elemType.getUpperBound() : elemType);
        }

        if (type instanceof WildcardType) {
            WildcardType wildcard = (WildcardType) type;
            // todo: easy optimise like above
            ReferenceType upper = subst(wildcard.getUpperBound(), mapper);
            ReferenceType lower = subst(wildcard.getLowerBound(), mapper);
            return new WildcardType(
                    upper instanceof WildcardType ? upper.getUpperBound() : upper,
                    lower instanceof WildcardType ? lower.getLowerBound() : lower);
        }

        if (type instanceof IntersectionType) {
            IntersectionType intersection = (IntersectionType) type;
            // todo: optimise
            return new IntersectionType(
                    intersection.baseType == null ? null : subst(intersection.baseType, mapper),
                    of(intersection.getInterfaces()).map(type1 -> subst(type1, mapper)).toImmutableList());
        }

        return subst((ClassType) type, mapper);
    }

    /**
     * Apply the substitutions given by the provided mapper recursively.
     *
     * @param type   The input type.
     * @param mapper The {@link TypeMapper}.
     * @return The substituted type.
     */
    public static ClassType subst(ClassType type, TypeMapper mapper) {
        if (type instanceof ParameterizedClass) {
            ParameterizedClass pClass = (ParameterizedClass) type;
            // todo: optimise if applying the binding function doesn't change anything
            return new ParameterizedClass(
                    pClass.getOuter() == null ? null : (ParameterizedClass) subst(pClass.getOuter(), mapper),
                    pClass.getDeclaration(),
                    of(pClass.getTypeArguments()).map(e -> subst(e, mapper)).toImmutableList());
        }

        return type;
    }

    /**
     * Parameterize the given field.
     *
     * @param owner The owner of the field. (where it is declared)
     * @param field The field to parameterize.
     * @return The parameterized field. May return the same field
     * passed in if no substitutions were made.
     */
    public static Field applyOwnerParameterization(ParameterizedClass owner, Field field) {
        if (field.isStatic()) return field;

        AType boundType = subst(field.getType(), owner);
        if (boundType == field.getType()) return field;

        return new ParameterizedField(owner, field, boundType);
    }

    public static Method applyOwnerParameterization(ParameterizedClass owner, Method method) {
        if (method.isStatic()) return method;

        assert method == method.getDeclaration();
        // TODO, dont return ParameterizedMethod if nothing was bound.
        return new ParameterizedMethod(owner, method, owner, false);
    }

    /**
     * Parameterize the given method.
     *
     * @param owner  The owner of the method. (where it is declared)
     * @param method The method to parameterize.
     * @param mapper The {@link TypeMapper} to apply.
     * @return The parameterized method.
     */
    public static Method parameterize(ClassType owner, Method method, TypeParamMapper mapper) {
        // TODO, dont return ParameterizedMethod if nothing was bound.
        return new ParameterizedMethod(owner, method, mapper, true);
    }

    public interface TypeMapper {

        /**
         * Optionally substitute every type.
         * <p>
         * May produce an identity mapping.
         *
         * @param type The type to substitute.
         * @return The substituted type.
         */
        ReferenceType mapType(ReferenceType type);

        default TypeSubstApplier substFunc() {
            return t -> subst(t, this);
        }
    }

    public interface TypeSubstApplier extends Function<ReferenceType, ReferenceType> {
        TypeSubstApplier NONE = e -> e;
    }

    public interface TypeParamMapper extends TypeMapper {

        @Override
        default ReferenceType mapType(ReferenceType type) {
            return type instanceof TypeParameter ? mapParam((TypeParameter) type) : type;
        }

        /**
         * Map the given {@link TypeParameter} to a {@link ReferenceType}.
         * <p>
         * May produce an identity mapping.
         *
         * @param param The parameter to map.
         * @return The result.
         */
        ReferenceType mapParam(TypeParameter param);
    }
}
