package net.covers1624.coffeegrinder.type.asm;

import net.covers1624.coffeegrinder.bytecode.AccessFlag;
import net.covers1624.coffeegrinder.type.*;
import net.covers1624.coffeegrinder.util.EnumBitSet;
import net.covers1624.coffeegrinder.util.Util;
import org.jetbrains.annotations.Nullable;
import org.objectweb.asm.Type;
import org.objectweb.asm.tree.FieldNode;

/**
 * Created by covers1624 on 13/4/21.
 */
public class AsmField extends Field {

    private final TypeResolver typeResolver;
    private final AsmClass owner;
    private final FieldNode fNode;
    private final Type desc;
    private final AType type;
    @Nullable
    private final AType genericType;

    private final EnumBitSet<AccessFlag> accessFlags;

    private final AnnotationSupplier annotationSupplier;

    AsmField(TypeResolver typeResolver, AsmClass owner, FieldNode fNode) {
        this.typeResolver = typeResolver;
        this.owner = owner;
        this.fNode = fNode;
        desc = Type.getType(fNode.desc);
        type = typeResolver.resolveType(desc);
        genericType = fNode.signature != null ? ReferenceTypeSignatureParser.parse(typeResolver, owner, fNode.signature) : null;

        accessFlags = AccessFlag.unpackField(fNode.access);
        annotationSupplier = new AnnotationSupplier(typeResolver,
                Util.safeConcat(fNode.visibleAnnotations, fNode.invisibleAnnotations),
                Util.safeConcat(fNode.visibleTypeAnnotations, fNode.invisibleTypeAnnotations)
        );
    }

    //@formatter:off
    public FieldNode getNode() { return fNode; }
    @Override public ClassType getDeclaringClass() { return owner; }
    @Override public EnumBitSet<AccessFlag> getAccessFlags() { return accessFlags; }
    @Override public String getName() { return fNode.name; }
    @Override public AType getType() { return genericType != null ? genericType : type; }
    @Override public Type getDescriptor() { return desc; }
    @Override public AnnotationSupplier getAnnotationSupplier() { return annotationSupplier; }
    @Override @Nullable public Object getConstantValue() { return fNode.value; }
    @Override public Field getDeclaration() { return this; }
    //@formatter:on

    @Override
    public Field asRaw() {
        return genericType != null ? new RawField(this, type) : this;
    }

}
