package net.covers1624.coffeegrinder.util.resolver;

import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;

import java.io.Closeable;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;

/**
 * A Resolver, capable of loading the raw bytes for a given Class or Resource.
 * <p>
 * Created by covers1624 on 8/4/21.
 */
public interface Resolver extends Closeable {

    /**
     * Gets all the classes available from this {@link Resolver}.
     *
     * @return The classes.
     */
    FastStream<String> getAllClasses() throws IOException;

    /**
     * Gets all the non-class files available from this {@link Resolver}.
     *
     * @return The non-class files.
     */
    FastStream<String> getAllResources() throws IOException;

    /**
     * Gets the bytes for a given class.
     *
     * @param name The class name.
     * @return The Class bytes, or literal null if no bytes were found.
     * @throws IOException If an error occurred whilst reading a file.
     * @see #getAllClasses()
     */
    default byte @Nullable [] getClassBytes(String name) throws IOException {
        return getResource(name.replace(".", "/") + ".class");
    }

    /**
     * Checks if this resolver is capable of resolving the given class name.
     *
     * @param name The class name.
     * @return If the resolver can resolve this class.
     */
    default boolean hasClass(String name) {
        return hasResource(name.replace('.', '/') + ".class");
    }

    /**
     * Gets the bytes for a given resource.
     *
     * @param name The name of the input.
     * @return The bytes of the resource or literal null if no bytes were found.
     * @throws IOException If an error occurred whilst reading a file.
     * @see #getAllResources()
     */
    byte @Nullable [] getResource(String name) throws IOException;

    /**
     * Checks if this resolver is capable of resolving the given resource.
     *
     * @param name The resource name.
     * @return If the resolver can resolve this class.
     */
    boolean hasResource(String name);

    @Override
    default void close() throws IOException { }

    /**
     * Finds a resolver for the given file.
     *
     * @param path The file path.
     * @return The {@link Resolver}.
     * @throws IllegalArgumentException Thrown if a resolver was not found for a given type.
     */
    static Resolver findResolver(Path path) {
        if (Files.isDirectory(path)) {
            return new FolderResolver(path);
        }
        String fName = path.getFileName().toString();
        if (fName.endsWith(".jar") || fName.endsWith(".zip")) {
            return new JarResolver(path);
        }
        if (fName.endsWith(".jmod")) {
            return new JModResolver(path);
        }
        throw new IllegalArgumentException("Unable to find Resolver for type.");
    }
}
