package net.covers1624.coffeegrinder.bytecode.transform.transformers;

import net.covers1624.coffeegrinder.bytecode.InsnOpcode;
import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.SimpleInsnVisitor;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.transform.MethodTransformContext;
import net.covers1624.coffeegrinder.bytecode.transform.MethodTransformer;
import net.covers1624.coffeegrinder.util.None;
import org.jetbrains.annotations.Nullable;

import static java.util.Objects.requireNonNull;
import static net.covers1624.coffeegrinder.bytecode.matching.BranchLeaveMatching.matchLeave;
import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.matchLoadLocal;
import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.matchStore;

/**
 * Transforms MonitorEnter/Exit + Try-Finally into a Synchronized block.
 * <p>
 * Created by covers1624 on 8/7/21.
 */
public class SynchronizedTransform extends SimpleInsnVisitor<MethodTransformContext> implements MethodTransformer {

    @Override
    public void transform(MethodDecl function, MethodTransformContext ctx) {
        function.accept(this, ctx);
    }

    @Override
    public None visitTryFinally(TryFinally tryFinally, MethodTransformContext ctx) {
        Monitor monitor = matchMonitorInsn(tryFinally.getPrevSiblingOrNull(), InsnOpcode.MONITOR_ENTER);
        if (monitor == null) return super.visitTryFinally(tryFinally, ctx);

        // MONITOR_ENTER(STORE finallyVar, ...)
        // TRY_FINALLY {
        //   ...
        // } finally {
        //   MONITOR_EXIT(LOAD finallyVar)
        //   LEAVE finally
        // }
        // ->
        // SYNCHRONIZED (...) {
        //   ...
        // }
        ctx.pushStep("Generate Synchronized");

        Store store = requireNonNull(matchStore(monitor.getArgument()));

        Block finallyBody = tryFinally.getFinallyBody().getEntryPoint();
        Monitor monitorExit = requireNonNull(matchMonitorInsn(finallyBody.getFirstChild(), InsnOpcode.MONITOR_EXIT));

        requireNonNull(matchLoadLocal(monitorExit.getArgument(), store.getVariable()));
        requireNonNull(matchLeave(monitorExit.getNextSibling(), tryFinally.getFinallyBody()));

        Synchronized sync = tryFinally.replaceWith(new Synchronized(store.getValue(), tryFinally.getTryBody()).withOffsets(tryFinally));

        assert store.getVariable().getLoadCount() == 0;
        monitor.remove();
        ctx.popStep();

        sync.accept(this, ctx);
        return NONE;
    }

    /**
     * Matches the given instruction to a MonitorInsn instruction.
     *
     * @param insn The insn to match against.
     * @return The MonitorInsn instruction.
     */
    @Nullable
    public static Monitor matchMonitorInsn(@Nullable Instruction insn, InsnOpcode opcode) {
        if (insn == null || opcode != insn.opcode) return null;
        return (Monitor) insn;
    }
}

