package net.covers1624.coffeegrinder.source;

import com.google.common.collect.ImmutableList;

import java.util.List;

/**
 * Created by covers1624 on 8/8/21.
 */
@SuppressWarnings ("UnstableApiUsage")
public class LineBuffer {

    public static final LineBuffer EMPTY = of(ImmutableList.of());

    public final ImmutableList<String> lines;

    private LineBuffer(List<String> lines) {
        this.lines = ImmutableList.copyOf(lines); // Does not copy if input is an ImmutableList.
    }

    public static LineBuffer of() {
        return EMPTY;
    }

    public static LineBuffer of(Object str) {
        return of(ImmutableList.of(str.toString()));
    }

    public static LineBuffer of(List<String> lines) {
        return new LineBuffer(lines);
    }

    public LineBuffer prepend(String prepend) {
        if (prepend.isEmpty()) return this;
        if (lines.size() == 1) return of(prepend + lines.get(0));

        ImmutableList.Builder<String> builder = new ImmutableList.Builder<>();
        builder.add(prepend + lines.get(0));
        builder.addAll(lines.subList(1, lines.size()));
        return of(builder.build());
    }

    public LineBuffer append(Object append) {
        String str = append.toString();
        if (lines.isEmpty()) return of(ImmutableList.of(str));
        if (lines.size() == 1) return of(lines.get(0) + str);
        if (str.isEmpty()) return this;

        ImmutableList.Builder<String> builder = ImmutableList.builderWithExpectedSize(lines.size());
        builder.addAll(lines.subList(0, lines.size() - 1));
        builder.add(lines.get(lines.size() - 1) + str);
        return of(builder.build());
    }

    public LineBuffer append(LineBuffer other) {
        return append("", other);
    }

    public LineBuffer append(String sep, LineBuffer other) {
        if (lines.isEmpty()) return other.prepend(sep);
        if (other.lines.isEmpty()) return append(sep);

        ImmutableList.Builder<String> builder = ImmutableList.builderWithExpectedSize(lines.size() + other.lines.size());

        builder.addAll(lines.subList(0, lines.size() - 1));
        builder.add(lines.get(lines.size() - 1) + sep + other.lines.get(0));
        builder.addAll(other.lines.subList(1, other.lines.size()));
        return of(builder.build());
    }

    public LineBuffer add(String line) {
        return of(ImmutableList.<String>builderWithExpectedSize(lines.size() + 1)
                .addAll(lines)
                .add(line)
                .build()
        );
    }

    public LineBuffer join(LineBuffer other) {
        if (lines.isEmpty()) return other;
        if (other.lines.isEmpty()) return this;

        return of(ImmutableList.<String>builderWithExpectedSize(lines.size() + other.lines.size())
                .addAll(lines)
                .addAll(other.lines)
                .build()
        );
    }

    public LineBuffer joinOn(String seperator) {
        return of(String.join(seperator, lines));
    }

    @Override
    public String toString() {
        return String.join("\n", lines);
    }
}
