package net.covers1624.coffeegrinder.type.asm;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import net.covers1624.coffeegrinder.type.*;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;
import org.objectweb.asm.AnnotationVisitor;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;
import org.objectweb.asm.tree.AnnotationNode;

import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;

/**
 * Created by covers1624 on 13/9/22.
 */
public abstract class AnnotationParser extends AnnotationVisitor {

    private final TypeResolver resolver;

    public AnnotationParser(TypeResolver resolver) {
        super(Opcodes.ASM9);
        this.resolver = resolver;
    }

    public static void parseNodes(TypeResolver resolver, TypeAnnotationData typeAnnotationConflicts, @Nullable Iterable<AnnotationNode> nodes, Consumer<AnnotationData> cons) {
        if (nodes == null) return;

        for (AnnotationNode node : nodes) {
            node.accept(newVisitor(resolver, node.desc, e -> {
                // If we have a type annotation of the exact same, exclude from the base.
                // We don't need to be mindful of repeatable annotations here, they are still flattened into
                // their container list annotation.
                if (!typeAnnotationConflicts.annotations.contains(e)) {
                    cons.accept(e);
                }
            }));
        }
    }

    public static AnnotationParser newVisitor(TypeResolver resolver, String desc, Consumer<AnnotationData> cons) {
        ClassType type = resolver.resolveClass(Type.getType(desc));
        ImmutableMap.Builder<String, Object> values = ImmutableMap.builder();

        return new AnnotationParser(resolver) {
            @Override
            public void visitValue(@Nullable String name, Object value) {
                assert name != null;
                values.put(name, value);
            }

            @Override
            public void visitEnd() {
                cons.accept(new AnnotationData(type, values.build()));
            }
        };
    }

    public static Object processAnnotationDefault(TypeResolver resolver, Object obj) {
        AtomicReference<Object> valueRef = new AtomicReference<>();
        AnnotationParser visitor = new AnnotationParser(resolver) {
            @Override
            public void visitValue(@Nullable String name, Object value) {
                assert name == null;
                valueRef.set(value);
            }
        };
        accept(visitor, obj);
        visitor.visitEnd();
        assert valueRef.get() != null;

        return valueRef.get();
    }

    public abstract void visitValue(@Nullable String name, Object value);

    @Override
    public void visit(@Nullable String name, Object value) {
        visitValue(name, processValue(resolver, value));
    }

    @Override
    public void visitEnum(String name, String descriptor, String value) {
        ClassType type = resolver.resolveClass(Type.getType(descriptor));
        Field field = FastStream.of(type.getFields())
                .filter(f -> f.getName().equals(value))
                .only();
        visitValue(name, field);
    }

    @Override
    public AnnotationVisitor visitAnnotation(String name, String descriptor) {
        return newVisitor(resolver, descriptor, data -> visitValue(name, data));
    }

    @Override
    public AnnotationVisitor visitArray(String name) {
        return new AnnotationParser(resolver) {
            private final ImmutableList.Builder<Object> values = ImmutableList.builder();

            @Override
            public void visitValue(@Nullable String name, Object value) {
                assert name == null;
                values.add(value);
            }

            @Override
            public void visitEnd() {
                AnnotationParser.this.visitValue(name, values.build());
            }
        };
    }

    private static Object processValue(TypeResolver resolver, Object value) {
        if (value instanceof Type) {
            // The java/objectweb docs say we will only ever get Classes and Array types here.
            AType result = resolver.resolveTypeDecl((Type) value);
            assert result instanceof ArrayType || result instanceof ClassType;
            return result;
        }
        // We can _theoretically_ get Primitive arrays here, but AnnotationNode never gives those to us.
        assert !value.getClass().isArray();
        return value;
    }

    // Mostly a copy of the method from AnnotationNode. Objectweb hides this away and gives us
    // no easy way to only visit the default annotation value of a MethodNode without
    // visiting the entire damn thing including all the code.
    private static void accept(AnnotationVisitor visitor, Object value) {
        if (value instanceof String[]) {
            String[] typeValue = (String[]) value;
            visitor.visitEnum(null, typeValue[0], typeValue[1]);
        } else if (value instanceof AnnotationNode) {
            AnnotationNode annotationValue = (AnnotationNode) value;
            annotationValue.accept(visitor.visitAnnotation(null, annotationValue.desc));
        } else if (value instanceof List) {
            AnnotationVisitor arrayAnnotationVisitor = visitor.visitArray(null);
            if (arrayAnnotationVisitor != null) {
                List<?> arrayValue = (List<?>) value;
                for (Object o : arrayValue) {
                    accept(arrayAnnotationVisitor, o);
                }
                arrayAnnotationVisitor.visitEnd();
            }
        } else {
            visitor.visit(null, value);
        }
    }
}
