package net.covers1624.coffeegrinder.bytecode.flow;

import guru.nidi.graphviz.attribute.Color;
import guru.nidi.graphviz.attribute.Rank;
import guru.nidi.graphviz.engine.Format;
import guru.nidi.graphviz.engine.Graphviz;
import guru.nidi.graphviz.model.Factory;
import guru.nidi.graphviz.model.Graph;
import guru.nidi.graphviz.model.Node;
import net.covers1624.coffeegrinder.bytecode.insns.Block;
import net.covers1624.quack.collection.FastStream;
import net.covers1624.quack.util.SneakyUtils;
import org.jetbrains.annotations.Nullable;

import java.io.File;
import java.util.*;
import java.util.function.Consumer;
import java.util.function.Function;

import static guru.nidi.graphviz.attribute.Rank.RankDir.TOP_TO_BOTTOM;
import static guru.nidi.graphviz.model.Factory.graph;

/**
 * Represents a block in the control flow graph.
 * <p>
 * Created by covers1624 on 28/2/21.
 */
public class ControlFlowNode {

    /**
     * The index in this nodes control flow graph.
     */
    public int cfgIndex;

    /**
     * The block this control flow node is for.
     */
    @Nullable
    public Block block;

    /**
     * Visited flag, used in various algorithms.
     */
    public boolean visited;

    /**
     * The index of the node in a post-order traversal of the control flow graph, starting at the
     * end point. This field gets computed by dominance analysis.
     */
    public int postOrderNumber;

    /**
     * List of incoming control flow edges.
     */
    private final List<ControlFlowNode> predecessors = new ArrayList<>();

    /**
     * List of outgoing control flow edges.
     */
    private final List<ControlFlowNode> successors = new ArrayList<>();

    /**
     * The immediate dominator (the parent in the dominator tree).
     * <code>null</code> if the dominance has not been calculated or the node is unreachable.
     */
    @Nullable
    ControlFlowNode immediateDominator;

    /**
     * List of children in the dominator tree.
     * <code>null</code> if the dominance has not been calculated or the node is unreachable.
     */
    @Nullable
    List<ControlFlowNode> dominatorTreeChildren = null;

    public ControlFlowNode() { }

    public ControlFlowNode(int cfgIndex) {
        this.cfgIndex = cfgIndex;
    }

    public ControlFlowNode(ControlFlowNode other) {
        this.cfgIndex = other.cfgIndex;
        this.block = other.block;
    }

    public ControlFlowNode(int cfgIndex, Block block) {
        this.cfgIndex = cfgIndex;
        this.block = block;
    }

    public void addEdgeTo(ControlFlowNode target) {
        successors.add(target);
        target.predecessors.add(this);
    }

    public void traversePreOrder(Function<ControlFlowNode, List<ControlFlowNode>> children, Consumer<ControlFlowNode> visitor) {
        if (visited) return;
        visited = true;
        visitor.accept(this);
        for (ControlFlowNode t : children.apply(this)) {
            t.traversePreOrder(children, visitor);
        }
    }

    public void traversePostOrder(Function<ControlFlowNode, List<ControlFlowNode>> children, Consumer<ControlFlowNode> visitor) {
        if (visited) return;
        visited = true;
        for (ControlFlowNode t : children.apply(this)) {
            t.traversePostOrder(children, visitor);
        }
        visitor.accept(this);
    }

    /**
     * Gets whether this node dominates the provided node.
     *
     * @param node The node to check against.
     * @return If this node dominates the provided node.
     */
    public boolean dominates(ControlFlowNode node) {
        ControlFlowNode tmp = node;
        while (tmp != null) {
            if (tmp == this) return true;

            tmp = tmp.immediateDominator;
        }
        return false;
    }

    public FastStream<ControlFlowNode> streamPreOrder(Function<ControlFlowNode, FastStream<ControlFlowNode>> children) {
        return FastStream.of(this).concat(children.apply(this).flatMap(e -> e.streamPreOrder(children)));
    }

    public FastStream<ControlFlowNode> streamPostOrder(Function<ControlFlowNode, FastStream<ControlFlowNode>> children) {
        return children.apply(this).flatMap(e -> e.streamPostOrder(children)).concat(FastStream.of(this));
    }

    /**
     * Gets the {@link #block}, requiring it to be non-null.
     * (mostly for nullability convenience.)
     *
     * @return The block.
     */
    public Block getBlock() {
        return Objects.requireNonNull(block);
    }

    /**
     * @return List of incoming control flow edges.
     */
    public List<ControlFlowNode> getPredecessors() {
        return predecessors;
    }

    /**
     * @return List of outgoing control flow edges.
     */
    public List<ControlFlowNode> getSuccessors() {
        return successors;
    }

    /**
     * Gets the immediate dominator (the parent in the dominator tree).
     * <code>null</code> if the dominance has not been calculated; or the node is unreachable.
     * <p>
     * This is an overload for {@link #getImmediateDominatorOrNull()}
     *
     * @return The immediate parent, or null.
     */
    public ControlFlowNode getImmediateDominator() {
        return Objects.requireNonNull(getImmediateDominatorOrNull());
    }

    /**
     * Gets the immediate dominator (the parent in the dominator tree).
     * <code>null</code> if the dominance has not been calculated; or the node is unreachable.
     *
     * @return The immediate parent, or null.
     */
    @Nullable
    public ControlFlowNode getImmediateDominatorOrNull() {
        return immediateDominator;
    }

    /**
     * List of children in the dominator tree.
     *
     * @return The children.
     */
    public List<ControlFlowNode> getDominatorTreeChildren() {
        return Objects.requireNonNull(dominatorTreeChildren);
    }

    /**
     * Gets whether this node is reachable. Requires that dominance is computed!
     *
     * @return If the node is reachable.
     */
    public boolean isReachable() {
        return dominatorTreeChildren != null;
    }

    @Override
    public String toString() {
        if (block == null) return "Block at: UNKNOWN";
        return "Block at: " + block.getName();
    }

    private Graph addToGraph(Graph g, Set<ControlFlowNode> visited, Function<ControlFlowNode, String> namer) {
        if (!visited.add(this)) { return g; }

        String name = namer.apply(this);
        Node node = Factory.node(name);
        if (immediateDominator != null) {
            g = g.with(Factory.node(namer.apply(immediateDominator)).link(Factory.to(node).with(Color.LIMEGREEN)));
        }

        for (ControlFlowNode n : successors) {
            g = n.addToGraph(g, visited, namer);
            node = node.link(namer.apply(n));
        }

        g = g.with(node);
        for (ControlFlowNode n : predecessors) {
            g = n.addToGraph(g, visited, namer);
        }
        return g;
    }

    public static Graph makeGraph(ControlFlowNode entry) {
        Graph graph = graph().directed()
                .graphAttr().with(Rank.dir(TOP_TO_BOTTOM))
                .linkAttr().with("class", "link-class");

        Set<ControlFlowNode> visited = new HashSet<>();
        Map<ControlFlowNode, String> unnamedNodes = new HashMap<>();
        graph = entry.addToGraph(graph, visited, e -> e.block != null ? e.block.getName() : unnamedNodes.computeIfAbsent(e, e2 -> "#" + unnamedNodes.size()));
        return graph;
    }

    // Exists for debugger evaluation.
    private void evalDumpGraph() {
        evalDumpGraph("eval_graph");
    }

    private void evalDumpGraph(String name) {
        SneakyUtils.sneaky(() -> Graphviz.fromGraph(makeGraph(this)).scale(2).render(Format.PNG).toFile(new File("graphs/" + name + ".png")));
    }

}
