package net.covers1624.coffeegrinder.bytecode.transform;

import com.google.common.collect.ImmutableList;
import net.covers1624.coffeegrinder.bytecode.InsnOpcode;
import net.covers1624.coffeegrinder.bytecode.InvariantVisitor;
import net.covers1624.coffeegrinder.bytecode.flow.ControlFlowGraph;
import net.covers1624.coffeegrinder.bytecode.flow.ControlFlowNode;
import net.covers1624.coffeegrinder.bytecode.insns.Block;
import net.covers1624.coffeegrinder.bytecode.insns.BlockContainer;
import net.covers1624.coffeegrinder.bytecode.insns.MethodDecl;

import java.util.List;

/**
 * An {@link MethodTransformer} that runs a list of per-block transforms.
 * <p>
 * Created by covers1624 on 19/4/21.
 */
public class MethodBlockTransform implements MethodTransformer {

    private final String name;
    private final List<BlockTransformer> transforms;

    private boolean running;

    /**
     * Constructs a new {@link MethodBlockTransform} representing a single list of transforms
     * visited in post-order.
     *
     * @param transforms The post-order transforms.
     */
    public MethodBlockTransform(String name, List<BlockTransformer> transforms) {
        this.name = name;
        this.transforms = ImmutableList.copyOf(transforms);
        assert !transforms.isEmpty();
    }

    public static MethodBlockTransform of(String name, BlockTransformer... transforms) {
        return new MethodBlockTransform(name, ImmutableList.copyOf(transforms));
    }

    @Override
    public void transform(MethodDecl function, MethodTransformContext ctx) {
        if (running) throw new IllegalStateException("Reentrancy detected.");

        try {
            running = true;
            ctx.pushTiming("Find BlockContainers");
            List<BlockContainer> blocks = function.descendantsToList(InsnOpcode.BLOCK_CONTAINER);
            ctx.popTiming();
            for (BlockContainer container : blocks) {
                ctx.pushTiming("Control Flow Graph");
                ControlFlowGraph graph = new ControlFlowGraph(container);
                ctx.popTiming();
                visitBlock(graph, graph.nodes.get(container.getEntryPointOrNull()), ctx);
            }

        } finally {
            running = false;
        }
    }

    private void visitBlock(ControlFlowGraph graph, ControlFlowNode cfgNode, MethodTransformContext ctx) {
        Block block = cfgNode.getBlock();

        for (ControlFlowNode child : cfgNode.getDominatorTreeChildren()) {
            visitBlock(graph, child, ctx);
        }

        BlockTransformContext blockCtx = new BlockTransformContext(ctx, cfgNode, graph);
        blockCtx.pushStep(block.getName());
        //noinspection ForLoopReplaceableByForEach
        for (int i = 0; i < transforms.size(); i++) {
            BlockTransformer t = transforms.get(i);
            blockCtx.pushStep(t.getName(), t.stepType());
            t.transform(block, blockCtx);
            InvariantVisitor.checkInvariants(block);
            blockCtx.popStep();
        }
        blockCtx.popStep();
    }

    @Override
    public String getName() {
        return name;
    }
}
