package net.covers1624.coffeegrinder.bytecode.transform.transformers;

import net.covers1624.coffeegrinder.bytecode.InsnOpcode;
import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.SimpleInsnVisitor;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.matching.AssignmentMatching;
import net.covers1624.coffeegrinder.bytecode.transform.MethodTransformContext;
import net.covers1624.coffeegrinder.bytecode.transform.MethodTransformer;
import net.covers1624.coffeegrinder.bytecode.transform.StatementTransformContext;
import net.covers1624.coffeegrinder.bytecode.transform.StatementTransformer;
import net.covers1624.coffeegrinder.type.TypeSystem;
import net.covers1624.coffeegrinder.util.None;

import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.matchLoad;
import static net.covers1624.coffeegrinder.bytecode.transform.transformers.statement.AssignmentExpressions.equivalentRefs;

/**
 * Created by covers1624 on 5/3/22.
 */
public class CompoundAssignments extends SimpleInsnVisitor<MethodTransformContext> implements StatementTransformer, MethodTransformer {

    @Override
    public void transform(Instruction statement, StatementTransformContext ctx) {
        statement.accept(this, ctx);
    }

    @Override
    public void transform(MethodDecl function, MethodTransformContext ctx) {
        function.accept(this, ctx);
    }

    @Override
    public None visitStore(Store store, MethodTransformContext ctx) {
        if (transformCompoundAssignment(store, ctx)) {
            return NONE;
        }

        return super.visitStore(store, ctx);
    }

    public boolean transformCompoundAssignment(Store store, MethodTransformContext ctx) {
        // STORE (ref, NUMERIC.op(LOAD ref, ...))
        // ->
        // COMPOUND_ASSIGNMENT.op(ref, ...)
        Binary binary = AssignmentMatching.matchStoreArgBinaryWithPossibleCast(store);
        if (binary == null) return false;

        // Find the most nested binary add on the left returning String.
        // Sometimes nested binary may exist without a String on the left when math is involved. (i / j + s)
        if (TypeSystem.isString(binary.getResultType())) {
            while (binary.getLeft().opcode == InsnOpcode.BINARY && TypeSystem.isString(binary.getLeft().getResultType())) {
                binary = (Binary) binary.getLeft();
            }
        }

        Load numericLoad = matchLoad(binary.getLeft());
        if (numericLoad == null) return false;
        if (!equivalentRefs(store.getReference(), numericLoad.getReference())) return false;

        ctx.pushStep("Create compound assignment");

        if (store.getValue().opcode == InsnOpcode.CHECK_CAST) {
            store.getValue().replaceWith(binary);
        }

        // Handles both nested and non nested compound assignment patterns.
        // Means we only ever need to use the store value bellow :)
        binary.replaceWith(binary.getRight());

        Instruction repl = new CompoundAssignment(binary.getOp(), store.getReference(), store.getValue());
        store.replaceWith(repl.withOffsets(store));
        ctx.popStep();
        return true;
    }
}
