package net.covers1624.coffeegrinder.bytecode.transform.transformers;

import net.covers1624.coffeegrinder.bytecode.AccessFlag;
import net.covers1624.coffeegrinder.bytecode.InsnOpcode;
import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.matching.InvokeMatching;
import net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching;
import net.covers1624.coffeegrinder.bytecode.transform.ClassTransformContext;
import net.covers1624.coffeegrinder.bytecode.transform.ClassTransformer;
import net.covers1624.coffeegrinder.type.ClassType;
import net.covers1624.coffeegrinder.type.Field;
import net.covers1624.coffeegrinder.type.Parameter;
import net.covers1624.coffeegrinder.type.TypeSystem;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;

import java.util.Objects;

import static com.google.common.collect.ImmutableList.copyOf;
import static java.util.Objects.requireNonNull;
import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.matchLoad;
import static net.covers1624.quack.util.SneakyUtils.notPossible;

/**
 * Created by covers1624 on 2/9/21.
 */
public class InnerClasses implements ClassTransformer {

    @SuppressWarnings ("NotNullFieldNotInitialized")
    private ClassTransformContext ctx;

    @SuppressWarnings ("NotNullFieldNotInitialized")
    private ClassType outerType;

    public void transform(ClassDecl cInsn, ClassTransformContext ctx) {
        this.ctx = ctx;

        cInsn.getClassMembers().forEach(e -> processInnerClass(e, ctx));
    }

    private void processInnerClass(ClassDecl cInsn, ClassTransformContext ctx) {
        if (cInsn.getClazz().isStatic()) return;

        ctx.pushStep("Remove synthetic Outer.this field");
        outerType = cInsn.getClazz().getEnclosingClass().orElseThrow(notPossible());

        FastStream<MethodDecl> constructors = cInsn.getMethodMembers().filter(f -> f.getMethod().isConstructor());

        Field thisField = constructors.map(this::matchOuterThisField).filter(Objects::nonNull).distinct().onlyOrDefault();
        assert thisField == null || ((ClassType) thisField.getType()).getDeclaration() == outerType;
        assert thisField == null || thisField.isSynthetic();

        constructors.forEach(this::removeOuterThisParam);

        if (thisField != null) {
            // LoadField(this$1)(LoadThis()) -> LoadThis(outer)
            cInsn.descendantsMatching(e -> LoadStoreMatching.matchLoadField(e, thisField))
                    .forEach(e -> e.replaceWith(new LoadThis(TypeSystem.makeThisType(outerType))));

            // remove the synthetic field
            cInsn.getFieldMembers().filter(f -> f.getField().equals(thisField)).only().remove();
        }

        ctx.popStep();
    }

    private void removeOuterThisParam(MethodDecl constructor) {
        // <init>(Outer arg0, ...)
        // STORE(FIELD this$1(LOAD_THIS(...)), LOAD arg0)
        // ->
        // <init>(...)

        Store storeField = LoadStoreMatching.matchStoreField(constructor.getBody().getEntryPoint().getFirstChild());
        if (storeField != null) storeField.remove();

        ParameterVariable thisParam = constructor.parameters.get(0);
        thisParam.makeImplicit();

        // there may be other references to the parameter (eg super constructor call)
        // super(Load arg0, ...) -> super(LoadThis(Outer), ...)
        copyOf(thisParam.getReferences())
                .forEach(e -> requireNonNull(matchLoad(e.getParent())).replaceWith(new LoadThis(TypeSystem.makeThisType(outerType))));
    }

    @Nullable
    private Field matchOuterThisField(MethodDecl constructor) {
        // <init>(Outer arg0, ...)
        // STORE(FIELD this$1(LOAD_THIS(...)), LOAD arg0)

        LocalVariable thisParam = constructor.parameters.get(0);
        Parameter parameter = constructor.getMethod().getParameters().get(0);
        assert ((ClassType) thisParam.getType()).getDeclaration() == outerType;

        Instruction first = constructor.getBody().getEntryPoint().getFirstChild();
        Store storeField = LoadStoreMatching.matchStoreField(first);
        if (storeField == null) {
            assert parameter.isMandated() || parameter.isSynthetic() || InvokeMatching.matchConstructorInvokeSpecial(first, constructor.getMethod().getDeclaringClass()) != null;
            return null;
        }
        FieldReference fieldRef = (FieldReference) storeField.getReference();
        assert fieldRef.getTarget().opcode == InsnOpcode.LOAD_THIS;
        assert LoadStoreMatching.matchLoadLocal(storeField.getValue(), thisParam) != null;
        return fieldRef.getField();
    }

}
