package net.covers1624.coffeegrinder.type;

import net.covers1624.coffeegrinder.type.asm.AsmClass;
import net.covers1624.coffeegrinder.type.asm.ReferenceTypeSignatureParser;
import net.covers1624.coffeegrinder.util.resolver.ClassResolver;
import org.jetbrains.annotations.Nullable;
import org.objectweb.asm.Type;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * Created by covers1624 on 12/4/21.
 */
public class TypeResolver {

    public static final Type OBJECT_TYPE = Type.getType(Object.class);
    public static final Type STRING_TYPE = Type.getType(String.class);
    public static final Type CLASS_TYPE = Type.getType(Class.class);
    public static final Type ENUM_TYPE = Type.getType(Enum.class);
    public static final Type THROWABLE_TYPE = Type.getType(Throwable.class);
    public static final Type SERIALIZABLE_TYPE = Type.getType(Serializable.class);
    public static final Type CLONEABLE_TYPE = Type.getType(Cloneable.class);

    private final ClassResolver classResolver;
    private final Map<Type, AType> typeCache = Collections.synchronizedMap(new HashMap<>());

    public TypeResolver(ClassResolver classResolver) {
        this.classResolver = classResolver;
    }

    public void reset() {
        typeCache.clear();
        classResolver.reset();
    }

    public AType resolveGenericType(ITypeParameterizedMember scope, String signature) {
        return ReferenceTypeSignatureParser.parse(this, scope, signature);
    }

    public AType resolveType(Type type) {
        AType ret = resolveTypeDecl(type);
        if (!(ret instanceof ClassType)) return ret;

        ClassType iClass = (ClassType) ret;
        if (TypeSystem.isGeneric(iClass)) {
            return iClass.asRaw();
        }

        return iClass;
    }

    public AType resolveTypeDecl(Type type) {
        // This can't be computeIfAbsent, because we modify typeCache.
        AType ret = typeCache.get(type);
        if (ret != null) return ret;
        synchronized (typeCache) {
            // Re-check, we may have waited on the synchronized for another thread to compute the value.
            ret = typeCache.get(type);
            if (ret != null) return ret;

            ret = computeType(type);
            AType prev = typeCache.put(type, ret);
            assert prev == null;
            return ret;
        }
    }

    private AType computeType(Type type) {
        switch (type.getSort()) {
            case Type.VOID:
                return PrimitiveType.VOID;
            case Type.BOOLEAN:
                return PrimitiveType.BOOLEAN;
            case Type.CHAR:
                return PrimitiveType.CHAR;
            case Type.BYTE:
                return PrimitiveType.BYTE;
            case Type.SHORT:
                return PrimitiveType.SHORT;
            case Type.INT:
                return PrimitiveType.INT;
            case Type.FLOAT:
                return PrimitiveType.FLOAT;
            case Type.LONG:
                return PrimitiveType.LONG;
            case Type.DOUBLE:
                return PrimitiveType.DOUBLE;
            case Type.ARRAY:
                return makeArray(resolveType(Type.getType(type.getDescriptor().substring(1))));
            case Type.OBJECT:
                return new AsmClass(this, classResolver.getClassNode(type.getInternalName()));
            default: {
                throw new IllegalStateException("Unhandled type. Sort: " + type.getSort());
            }
        }
    }

    public ReferenceType resolveReferenceType(Type type) {
        assert type.getSort() == Type.OBJECT || type.getSort() == Type.ARRAY;
        return (ReferenceType) resolveType(type);
    }

    public ClassType resolveClass(Class<?> clazz) {
        return resolveClass(Type.getType(clazz));
    }

    public ClassType resolveClass(String name) {
        return resolveClass(Type.getObjectType(name.replace('.', '/')));
    }

    public ClassType resolveClass(Type desc) {
        assert desc.getSort() == Type.OBJECT;

        return (ClassType) resolveType(desc);
    }

    public ClassType resolveClassDecl(String name) {
        return resolveClassDecl(Type.getObjectType(name.replace('.', '/')));
    }

    @Nullable
    public ClassType tryResolveClassDecl(String name) {
        try {
            return resolveClassDecl(name);
        } catch (ClassResolver.ClassNotFoundException ex) {
            return null;
        }
    }

    public ClassType resolveClassDecl(Type desc) {
        assert desc.getSort() == Type.OBJECT;

        return (ClassType) resolveTypeDecl(desc);
    }

    public ArrayType makeArray(AType elementType) {
        return new ArrayType(this, elementType);
    }

    public boolean classExists(String name) {
        return classResolver.classExists(name);
    }

    public ClassResolver getClassResolver() {
        return classResolver;
    }
}
