package net.covers1624.coffeegrinder.bytecode;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import guru.nidi.graphviz.engine.Format;
import guru.nidi.graphviz.engine.Graphviz;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.insns.tags.IIncTag;
import net.covers1624.coffeegrinder.type.*;
import net.covers1624.coffeegrinder.type.asm.AsmMethod;
import net.covers1624.coffeegrinder.util.None;
import net.covers1624.coffeegrinder.util.OpcodeLookup;
import net.covers1624.coffeegrinder.util.Util;
import net.covers1624.coffeegrinder.util.asm.AsmUtils;
import net.covers1624.coffeegrinder.util.asm.NodeAwareMethodVisitor;
import net.covers1624.quack.collection.FastStream;
import net.covers1624.quack.util.SneakyUtils;
import org.apache.commons.lang3.NotImplementedException;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.objectweb.asm.ConstantDynamic;
import org.objectweb.asm.Handle;
import org.objectweb.asm.Label;
import org.objectweb.asm.Type;
import org.objectweb.asm.tree.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.IntStream;

import static java.util.Objects.requireNonNull;
import static net.covers1624.coffeegrinder.DecompilerSettings.ASSERTIONS_ENABLED;
import static net.covers1624.coffeegrinder.bytecode.insns.LocalVariable.VariableKind;
import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.matchStoreLocal;
import static net.covers1624.coffeegrinder.type.TypeSystem.isAssignableTo;
import static net.covers1624.quack.collection.FastStream.of;
import static org.objectweb.asm.Opcodes.*;

/**
 * Created by covers1624 on 23/2/21.
 */
public class InstructionReader extends NodeAwareMethodVisitor {

    //region Reader static lookups.
    private static final int[] I_CONST = new int[] {
            -1,
            0,
            1,
            2,
            3,
            4,
            5
    };

    private static final Type[] A_TYPES = new Type[] {
            Type.INT_TYPE,
            Type.LONG_TYPE,
            Type.FLOAT_TYPE,
            Type.DOUBLE_TYPE,
            TypeResolver.OBJECT_TYPE,
            Type.BYTE_TYPE,// Byte/Boolean, Char, and Short are always loaded as Ints
            Type.CHAR_TYPE,
            Type.SHORT_TYPE,
    };

    private static final Type[] ILFDA_TYPES = new Type[] {
            Type.INT_TYPE,
            Type.LONG_TYPE,
            Type.FLOAT_TYPE,
            Type.DOUBLE_TYPE,
            TypeResolver.OBJECT_TYPE,
    };

    private static final Type[] NUMERIC_RESULTS = new Type[] {
            Type.INT_TYPE,
            Type.LONG_TYPE,
            Type.FLOAT_TYPE,
            Type.DOUBLE_TYPE,
            Type.INT_TYPE,
            Type.LONG_TYPE,
            Type.FLOAT_TYPE,
            Type.DOUBLE_TYPE,
            Type.INT_TYPE,
            Type.LONG_TYPE,
            Type.FLOAT_TYPE,
            Type.DOUBLE_TYPE,
            Type.INT_TYPE,
            Type.LONG_TYPE,
            Type.FLOAT_TYPE,
            Type.DOUBLE_TYPE,
            Type.INT_TYPE,
            Type.LONG_TYPE,
            Type.FLOAT_TYPE,
            Type.DOUBLE_TYPE,
            Type.INT_TYPE,
            Type.LONG_TYPE,
            Type.FLOAT_TYPE,
            Type.DOUBLE_TYPE,
            Type.INT_TYPE,
            Type.LONG_TYPE,
            Type.INT_TYPE,
            Type.LONG_TYPE,
            Type.INT_TYPE,
            Type.LONG_TYPE,
            Type.INT_TYPE,
            Type.LONG_TYPE,
            Type.INT_TYPE,
            Type.LONG_TYPE,
            Type.INT_TYPE,
            Type.LONG_TYPE,
    };

    private static final BinaryOp[] NUMERIC_OPS = new BinaryOp[] {
            BinaryOp.ADD,
            BinaryOp.ADD,
            BinaryOp.ADD,
            BinaryOp.ADD,
            BinaryOp.SUB,
            BinaryOp.SUB,
            BinaryOp.SUB,
            BinaryOp.SUB,
            BinaryOp.MUL,
            BinaryOp.MUL,
            BinaryOp.MUL,
            BinaryOp.MUL,
            BinaryOp.DIV,
            BinaryOp.DIV,
            BinaryOp.DIV,
            BinaryOp.DIV,
            BinaryOp.REM,
            BinaryOp.REM,
            BinaryOp.REM,
            BinaryOp.REM,
            null,
            null,
            null,
            null,
            BinaryOp.SHIFT_LEFT,
            BinaryOp.SHIFT_LEFT,
            BinaryOp.SHIFT_RIGHT,
            BinaryOp.SHIFT_RIGHT,
            BinaryOp.LOGICAL_SHIFT_RIGHT,
            BinaryOp.LOGICAL_SHIFT_RIGHT,
            BinaryOp.AND,
            BinaryOp.AND,
            BinaryOp.OR,
            BinaryOp.OR,
            BinaryOp.XOR,
            BinaryOp.XOR,
    };

    public static final Instruction[] NEG_LDC = new Instruction[] {
            new LdcNumber(0),
            new LdcNumber(0L),
            new LdcNumber(0.0F),
            new LdcNumber(0.0D),
    };

    public static final Type[][] PRIMITIVE_CAST = new Type[][] {
            { Type.INT_TYPE, Type.LONG_TYPE },
            { Type.INT_TYPE, Type.FLOAT_TYPE },
            { Type.INT_TYPE, Type.DOUBLE_TYPE },
            { Type.LONG_TYPE, Type.INT_TYPE },
            { Type.LONG_TYPE, Type.FLOAT_TYPE },
            { Type.LONG_TYPE, Type.DOUBLE_TYPE },
            { Type.FLOAT_TYPE, Type.INT_TYPE },
            { Type.FLOAT_TYPE, Type.LONG_TYPE },
            { Type.FLOAT_TYPE, Type.DOUBLE_TYPE },
            { Type.DOUBLE_TYPE, Type.INT_TYPE },
            { Type.DOUBLE_TYPE, Type.LONG_TYPE },
            { Type.DOUBLE_TYPE, Type.FLOAT_TYPE },
            { Type.INT_TYPE, Type.BYTE_TYPE },
            { Type.INT_TYPE, Type.CHAR_TYPE },
            { Type.INT_TYPE, Type.SHORT_TYPE },
    };

    public static final Type[] COMPARISONS = new Type[] {
            Type.LONG_TYPE,
            Type.FLOAT_TYPE,
            Type.FLOAT_TYPE,
            Type.DOUBLE_TYPE,
            Type.DOUBLE_TYPE,
    };

    public static final Compare.Kind[] CMP_EXT_KINDS = new Compare.Kind[] {
            Compare.Kind.LONG,
            Compare.Kind.NAN_L,
            Compare.Kind.NAN_G,
            Compare.Kind.NAN_L,
            Compare.Kind.NAN_G,
    };

    public static final Comparison.ComparisonKind[] IF_CMP_KINDS = new Comparison.ComparisonKind[] {
            Comparison.ComparisonKind.EQUAL,
            Comparison.ComparisonKind.NOT_EQUAL,
            Comparison.ComparisonKind.LESS_THAN,
            Comparison.ComparisonKind.GREATER_THAN_EQUAL,
            Comparison.ComparisonKind.GREATER_THAN,
            Comparison.ComparisonKind.LESS_THAN_EQUAL,
            Comparison.ComparisonKind.EQUAL,
            Comparison.ComparisonKind.NOT_EQUAL,
            Comparison.ComparisonKind.LESS_THAN,
            Comparison.ComparisonKind.GREATER_THAN_EQUAL,
            Comparison.ComparisonKind.GREATER_THAN,
            Comparison.ComparisonKind.LESS_THAN_EQUAL,
            Comparison.ComparisonKind.EQUAL,
            Comparison.ComparisonKind.NOT_EQUAL,
    };

    public static final Type[] T_TYPES = new Type[] {
            Type.BOOLEAN_TYPE,
            Type.CHAR_TYPE,
            Type.FLOAT_TYPE,
            Type.DOUBLE_TYPE,
            Type.BYTE_TYPE,
            Type.SHORT_TYPE,
            Type.INT_TYPE,
            Type.LONG_TYPE,
    };

    public static final Invoke.InvokeKind[] INVOKE_KINDS = new Invoke.InvokeKind[] {
            Invoke.InvokeKind.VIRTUAL,
            Invoke.InvokeKind.SPECIAL,
            Invoke.InvokeKind.STATIC,
            Invoke.InvokeKind.INTERFACE,
    };

    public static final Invoke.InvokeKind[] DYNAMIC_INVOKE_KINDS = new Invoke.InvokeKind[] {
            Invoke.InvokeKind.VIRTUAL,
            Invoke.InvokeKind.STATIC,
            Invoke.InvokeKind.SPECIAL,
            Invoke.InvokeKind.SPECIAL,
            Invoke.InvokeKind.INTERFACE
    };
    //endregion

    private static final Logger LOGGER = LoggerFactory.getLogger(InstructionReader.class);

    private final TypeResolver typeResolver;

    private final Method method;
    private final MethodNode mNode;

    private final int firstLocalIndex;
    private final ParameterVariable[] parameterVariables;
    private final List<ParameterVariable> paramVariablesList;

    private final MethodDecl function;
    private final BlockContainer mainContainer;

    @Nullable
    private final Label labelAfterCode;

    private final Object2IntMap<Label> importantLabels;
    private final Map<TryCatchBlockNode, TryCatch> tryCatches;
    private final Map<Label, Block> blockMap = new HashMap<>();

    private final Supplier<ClassType> thisType;

    private BlockContainer currentContainer;
    @Nullable
    private Block currentBlock;

    private int stackVarCounter;
    private final LinkedList<LocalVariable> currentStack = new LinkedList<>();
    private final LinkedList<LocalVariableNode> activeLVNodes = new LinkedList<>();

    private final VariableLivenessGraph vlGraph;

    private int currentIndex; // The current reader index in the MethodNode.
    private int sourceLineNumber = -1;

    private int variableCounter = 0;

    private InstructionReader(TypeResolver typeResolver, Method method) {
        super(ASM7);
        this.typeResolver = typeResolver;
        this.method = method;
        mNode = ((AsmMethod) method).getNode();

        firstLocalIndex = AsmUtils.getFirstLocalIndex(mNode);
        parameterVariables = computeParameters();
        paramVariablesList = FastStream.of(parameterVariables)
                .filter(Objects::nonNull)
                .toImmutableList();

        mainContainer = new BlockContainer();
        currentContainer = mainContainer;

        function = new MethodDecl(method, mainContainer, paramVariablesList);
        function.setReturnType(method.asRaw().getReturnType());
        function.addRef();

        AbstractInsnNode last = mNode.instructions.getLast();
        labelAfterCode = last instanceof LabelNode ? ((LabelNode) last).getLabel() : null;
        importantLabels = computeImportantLabels(mNode);

        vlGraph = new VariableLivenessGraph(mNode.maxLocals, firstLocalIndex, getBlock(requireNonNull(getFirstLabel(mNode))));
        tryCatches = generateTryCatchInsns(mNode.tryCatchBlocks);

        thisType = Util.singleMemoize(() -> TypeSystem.makeThisType(method.getDeclaringClass()));
    }

    private Map<TryCatchBlockNode, TryCatch> generateTryCatchInsns(List<TryCatchBlockNode> tryCatchBlocks) {
        Map<Label, TryCatch> handlerMap = new HashMap<>();
        AtomicInteger i = new AtomicInteger();

        return FastStream.of(tryCatchBlocks).toMap(Function.identity(), tcBlock -> {
            LocalVariable variable = new LocalVariable(
                    VariableKind.STACK_SLOT,
                    typeResolver.resolveType(tcBlock.type == null ? TypeResolver.THROWABLE_TYPE : Type.getObjectType(tcBlock.type)),
                    "e_" + i.getAndIncrement());

            Label handler = tcBlock.handler.getLabel();

            TryCatch tryCatch = handlerMap.computeIfAbsent(handler, x -> createTryCatch(handler, variable, tcBlock.type == null));
            LocalVariable var = tryCatch.handlers.only().getVariable().variable;
            if (var == variable) {
                function.variables.add(variable);
            } else {
                var.setType(TypeSystem.makeMultiCatchUnion((ReferenceType) var.getType(), (ReferenceType) variable.getType()));
            }

            vlGraph.addExceptionHandler(getBlock(handler), var);
            return tryCatch;
        });
    }

    @NotNull
    private TryCatch createTryCatch(Label handler, LocalVariable variable, boolean isFinally) {
        TryCatch tc = new TryCatch(new BlockContainer());
        BlockContainer handlerBody = new BlockContainer();
        Block handlerBlock = new Block();
        handlerBlock.instructions.add(new Branch(getBlock(handler)));
        handlerBody.blocks.add(handlerBlock);
        TryCatch.TryCatchHandler tcHandler = new TryCatch.TryCatchHandler(handlerBody, new LocalReference(variable));
        if (isFinally) {
            tcHandler.isUnprocessedFinally = true;
        }
        tc.handlers.add(tcHandler);
        return tc;
    }

    private ParameterVariable[] computeParameters() {
        ParameterVariable[] parameterVariables = new ParameterVariable[firstLocalIndex];

        int lIndex = !method.isStatic() ? 1 : 0;
        int pIndex = 0;
        for (Parameter parameter : method.getParameters()) {
            // Find variables for method parameters.
            ParameterVariable pVar = getParameterVariable(parameter, lIndex, pIndex++);
            parameterVariables[lIndex++] = pVar;
            if (parameter.getRawType() == PrimitiveType.LONG || parameter.getRawType() == PrimitiveType.DOUBLE) {
                parameterVariables[lIndex++] = null;
            }
        }
        return parameterVariables;
    }

    private ParameterVariable getParameterVariable(Parameter parameter, int index, int pIndex) {
        LocalVariableNode found = null;
        //Abstract methods will have no localVariables
        if (mNode.localVariables != null) {
            for (LocalVariableNode lv : mNode.localVariables) {
                if (lv.index == index) {
                    assert found == null : "Found duplicate LocalVariableNode for parameter index " + index;
                    found = lv;
                }
            }
        }
        assert found == null || parameter.getRawType().equals(typeResolver.resolveType(Type.getType(found.desc)));

        return new ParameterVariable(
                parameter,
                parameter.getRawType(),
                found != null ? found.signature : null, // TODO Parameter also has a copy of the signature..
                index,
                found != null ? found.name : "par_" + index,
                pIndex
        );
    }

    private LocalVariable getStoreLocal(int index, AType storingType) {
        // Parameters are special and will never be merged.
        if (index < firstLocalIndex) {
            return requireNonNull(parameterVariables[index]);
        }

        // Just generate a synthetic variable, keeping the 'potential' type intact.
        LocalVariable var = new LocalVariable(VariableKind.LOCAL, storingType, null, index, "var_" + variableCounter++);
        function.variables.add(var);
        var.setSynthetic(true);
        return var;
    }

    private LocalReference getLoadLocal(int index) {
        // Parameters are special and will never be merged.
        if (index < firstLocalIndex) {
            return new LocalReference(requireNonNull(parameterVariables[index]));
        }

        return vlGraph.readLocal(index);
    }

    private Block getBlock(Label label) {
        return blockMap.computeIfAbsent(label, e -> new Block(getBlockName(e)));
    }

    private String getBlockName(Label label) {
        int index = importantLabels.getOrDefault(label, -1);
        if (index != -1) return "L" + index;

        String name = requireNonNull(currentBlock).getName();
        int lastUnderscore = name.lastIndexOf("_");
        if (lastUnderscore != -1) {
            String num = name.substring(lastUnderscore + 1);
            if (StringUtils.isNumeric(num)) {
                return name.substring(0, lastUnderscore + 1) + (Integer.parseInt(num) + 1);
            }
        }
        return currentBlock.getSubName("1");
    }

    /**
     * Completely parse, process, and build a {@link MethodDecl}.
     *
     * @return The {@link MethodDecl}.
     */
    public static MethodDecl parse(TypeResolver typeResolver, Method method) {
        if (method.isAbstract()) {
            // TODO, Transform to find overrides and infer these names.
            // We can't generate these from LocalVariable information.
            int index = 0;
            List<ParameterVariable> parameters = new LinkedList<>();
            for (Parameter parameter : method.getParameters()) {
                parameters.add(new ParameterVariable(parameter, parameter.getType(), null, index, parameter.getName(), index++));
            }
            MethodDecl func = new MethodDecl(method, new Nop(), parameters);
            func.addRef();
            return func;
        }
        InstructionReader reader = new InstructionReader(typeResolver, method);

        reader.mNode.accept(reader);

        Set<String> parameterNames = ASSERTIONS_ENABLED ? FastStream.of(reader.paramVariablesList)
                .map(LocalVariable::getName)
                .toSet() : Collections.emptySet();
        Object2IntMap<String> variableNameUsages = new Object2IntOpenHashMap<>();
        for (LocalVariable v : reader.function.variables) {
            if (v.getKind() == VariableKind.LOCAL) {
                assert !parameterNames.contains(v.getName());
                v.setSubId(variableNameUsages.computeInt(v.getName(), (s, i) -> i == null ? 0 : i + 1));
            }
        }
        return reader.function;
    }

    // Exists for debugger evaluation.
    private void evalDumpGraph() {
        evalDumpGraph("eval_graph");
    }

    private void evalDumpGraph(String name) {
        SneakyUtils.sneaky(() -> Graphviz.fromGraph(vlGraph.makeGraph()).scale(2).render(Format.PNG).toFile(new File("graphs/" + name + ".png")));
    }

    /**
     * Push an instruction to the Stack.
     *
     * @param insn The instruction to push.
     * @return The Store Instruction for the stack Variable.
     */
    private Instruction push(Instruction insn, Type expectedStackType) {
        assertStackType(insn, expectedStackType);
        return pushUnchecked(insn);
    }

    private Instruction pushUnchecked(Instruction insn) {
        LocalVariable v = new LocalVariable(VariableKind.STACK_SLOT, insn.getResultType(), "s_" + stackVarCounter++);
        currentStack.push(v);
        function.variables.add(v);
        return new Store(new LocalReference(v), insn);
    }

    /**
     * Pop a value from the stack.
     * <p>
     * This method will remove the value from the stack.
     *
     * @return The Load Instruction for the stack Variable.
     */
    private Instruction pop() {
        if (currentStack.isEmpty()) throw new RuntimeException("Stack underflow.");
        return new Load(new LocalReference(currentStack.pop()));
    }

    /**
     * Pops a number of variables from the stack.
     * <p>
     * The same as {@link #pop()}, however, returns a number of entries preserving stack order.
     * <pre>
     *  Stack:
     *  - A
     *  - B
     *  - C
     *  - D
     *  - E
     *  ->
     *  insn = popMany(3);
     *  insn[0] == C
     *  insn[1] == D
     *  insn[2] == E
     * </pre>
     *
     * @param num The number of values to pop.
     * @return The List of Load instructions.
     */
    private List<Instruction> popMany(int num) {
        Instruction[] insns = new Instruction[num];
        for (int i = num - 1; i >= 0; i--) {
            insns[i] = pop();
        }
        return ImmutableList.copyOf(insns);
    }

    /**
     * Add a parsed instruction to the instructions list.
     *
     * @param insn The instruction to add.
     */
    private void addInsn(Instruction insn) {
        assert currentBlock != null;
        currentBlock.instructions.add(insn);
        insn.getDescendants().forEach(e -> {
            e.setBytecodeOffset(currentIndex);
            e.setSourceLine(sourceLineNumber);
        });
        if (ASSERTIONS_ENABLED) {
            insn.accept(new ReaderInvariantVisitor());
        }

        insn.<Branch>descendantsOfType(InsnOpcode.BRANCH).forEach(b -> {
            vlGraph.addCFEdge(b.getTargetBlock(), currentStack);
            adjustBranchTarget(b);
        });

        if (insn.getFlags().get(InstructionFlag.END_POINT_UNREACHABLE)) {
            currentStack.clear();
            currentBlock = null;
        } else if (insn.getFlags().get(InstructionFlag.MAY_BRANCH)) {
            // Split block after any instruction which may branch.
            visitImportantLabel(new Label());
        }

        // If this variable is a local store, we need to push a CFNode.
        Store store = matchStoreLocal(insn);
        if (store != null && store.getVariable().getKind() == VariableKind.LOCAL) {
            vlGraph.visitStore(store);

            LocalVariableNode lv = of(activeLVNodes).filter(n -> n.index == store.getVariable().getIndex()).onlyOrDefault();
            if (lv != null) {
                applyLVInfo(lv);
            }
        }

        // any active try-range handlers need an edge back to this point for variable lookup.
        // in theory, it's semantically valid to only add edges when the instruction may throw, and then only if the thrown type matches the handler
        // however, because both the try, and finally handler get their own copy of the finally body, the finally handler needs to 'start'
        // with all the same variable definitions that the try body 'ends' with. Source-wise, it's easier to just always link every cf node
        // Potentially, we could only make extra links to unprocessed finally handlers, and only make them when processing a branch which leaves a try range
        // however, I don't think being a little less aggressive with splitting around handlers will hurt us at all.
        for (TryCatch tryCatch : currentContainer.<TryCatch>ancestorsOfType(InsnOpcode.TRY_CATCH)) {
            vlGraph.addExceptionLink(getHandlerBlock(tryCatch));
        }
    }

    private void adjustBranchTarget(Branch b) {
        while (b.isDescendantOf(b.getTargetBlock()) && b.getTargetBlock() != currentBlock) {
            TryCatch tryCatch = (TryCatch) b.getTargetBlock().getFirstChild();
            b.setTargetBlock(tryCatch.getTryBody().getEntryPoint());
        }
    }

    @Override
    public void visitEnd() {
        vlGraph.applyAllReplacements(mainContainer);
        fixupOverlappingTryCatchHandlers();
    }

    private void fixupOverlappingTryCatchHandlers() {
        mainContainer.accept(new SimpleInsnVisitor<None>() {
            @Override
            public None visitTryCatch(TryCatch tryCatch, None ctx) {
                Block handler = getHandlerBlock(tryCatch);
                if (handler.isDescendantOf(tryCatch)) {
                    tryCatch.getParent().insertAfter(handler);
                }
                return super.visitTryCatch(tryCatch, ctx);
            }
        });
    }

    @Override
    public void visitInsn(AbstractInsnNode node, int index) {
        currentIndex = index;
        if (currentBlock != null || node.getType() == AbstractInsnNode.LABEL) {
            super.visitInsn(node, index);
        }
    }

    @Override
    public void visitInsn(int opcode) {
        String insnName = OpcodeLookup.getName(opcode);
        switch (opcode) {
            case NOP:
                break;
            case ACONST_NULL:
                addInsn(push(new LdcNull(), TypeResolver.OBJECT_TYPE));
                break;
            case ICONST_M1:
            case ICONST_0:
            case ICONST_1:
            case ICONST_2:
            case ICONST_3:
            case ICONST_4:
            case ICONST_5:
                addInsn(push(new LdcNumber(I_CONST[opcode - ICONST_M1]), Type.INT_TYPE));
                break;
            case LCONST_0:
                addInsn(push(new LdcNumber(0L), Type.LONG_TYPE));
                break;
            case LCONST_1:
                addInsn(push(new LdcNumber(1L), Type.LONG_TYPE));
                break;
            case FCONST_0:
                addInsn(push(new LdcNumber(0F), Type.FLOAT_TYPE));
                break;
            case FCONST_1:
                addInsn(push(new LdcNumber(1F), Type.FLOAT_TYPE));
                break;
            case FCONST_2:
                addInsn(push(new LdcNumber(2F), Type.FLOAT_TYPE));
                break;
            case DCONST_0:
                addInsn(push(new LdcNumber(0D), Type.DOUBLE_TYPE));
                break;
            case DCONST_1:
                addInsn(push(new LdcNumber(1D), Type.DOUBLE_TYPE));
                break;
            case IALOAD:
            case LALOAD:
            case FALOAD:
            case DALOAD:
            case AALOAD:
            case BALOAD:
            case CALOAD:
            case SALOAD: {
                Instruction index = pop();
                Instruction array = pop();
                addInsn(push(new Load(new ArrayElementReference(array, index)), A_TYPES[opcode - IALOAD]));
                break;
            }
            case IASTORE:
            case LASTORE:
            case FASTORE:
            case DASTORE:
            case AASTORE:
            case BASTORE:
            case CASTORE:
            case SASTORE: {
                Instruction value = pop();
                Instruction index = pop();
                Instruction array = pop();

                assertStackType(value, A_TYPES[opcode - IASTORE]);

                addInsn(new Store(new ArrayElementReference(array, index), value));
                break;
            }
            case POP:
                LocalVariable v = currentStack.pop();
                assert !isWide(v);
                break;
            case POP2:
                // POP2 will pop 2 non-wide variables, or a single wide variable.
                if (!isWide(currentStack.pop())) {
                    LocalVariable second = currentStack.pop();
                    // The second result must never be wide.
                    assert !isWide(second);
                }
                break;
            case DUP:
                currentStack.push(currentStack.peek());
                break;
            // TODO, add test cases for all of these DUP's
            case DUP_X1: {
                LocalVariable v1 = currentStack.pop();
                LocalVariable v2 = currentStack.pop();
                assert !isWide(v1);
                assert !isWide(v2);

                currentStack.push(v1);
                currentStack.push(v2);
                currentStack.push(v1);
                break;
            }
            case DUP_X2: {
                LocalVariable v1 = currentStack.pop();
                LocalVariable v2 = currentStack.pop();
                if (isWide(v2)) {
                    // Form 2
                    // v2, v1 -> v1, v2, v1
                    assert !isWide(v1);
                    assert isWide(v2);
                    currentStack.push(v1);
                    currentStack.push(v2);
                    currentStack.push(v1);
                    break;
                } else {
                    // Form 1
                    // v3, v2, v1 -> v1, v3, v2, v1
                    LocalVariable v3 = currentStack.pop();
                    assert !isWide(v1);
                    assert !isWide(v2);
                    assert !isWide(v3);
                    currentStack.push(v1);
                    currentStack.push(v3);
                    currentStack.push(v2);
                    currentStack.push(v1);
                    break;
                }
            }
            case DUP2: {
                LocalVariable v1 = currentStack.pop();
                if (isWide(v1)) {
                    // Form 2
                    // v1 -> v1, v1
                    assert isWide(v1);
                    currentStack.push(v1);
                    currentStack.push(v1);
                    break;
                } else {
                    // Form 1
                    // v2, v1 -> v2, v1, v2, v1
                    LocalVariable v2 = currentStack.pop();
                    assert !isWide(v1);
                    assert !isWide(v2);
                    currentStack.push(v2);
                    currentStack.push(v1);
                    currentStack.push(v2);
                    currentStack.push(v1);
                    break;
                }
            }
            case DUP2_X1: {
                LocalVariable v1 = currentStack.pop();
                LocalVariable v2 = currentStack.pop();

                if (isWide(v1)) {
                    // Form 2
                    // v2, v1 -> v1, v2, v1
                    assert isWide(v1);
                    assert !isWide(v2);
                    currentStack.push(v1);
                    currentStack.push(v2);
                    currentStack.push(v1);
                    break;
                } else {
                    // Form 1
                    // v3, v2, v1 -> v2, v1, v3, v2, v1
                    LocalVariable v3 = currentStack.pop();
                    assert !isWide(v1);
                    assert !isWide(v2);
                    assert !isWide(v3);
                    currentStack.push(v2);
                    currentStack.push(v1);
                    currentStack.push(v3);
                    currentStack.push(v2);
                    currentStack.push(v1);
                    break;
                }
            }
            case DUP2_X2: {
                LocalVariable v1 = currentStack.pop();
                LocalVariable v2 = currentStack.pop();

                if (!isWide(v1)) {
                    // Form 1/3
                    assert !isWide(v2);
                    LocalVariable v3 = currentStack.pop();
                    if (isWide(v3)) {
                        // Form 3
                        // v3, v2, v1 -> v2, v1, v3, v2, v1
                        assert !isWide(v1);
                        assert !isWide(v2);
                        assert isWide(v3);
                        currentStack.push(v2);
                        currentStack.push(v1);
                        currentStack.push(v3);
                        currentStack.push(v2);
                        currentStack.push(v1);
                        break;
                    } else {
                        // Form 1
                        // v4, v3, v2, v1 -> v2, v1, v4, v3, v2, v1
                        LocalVariable v4 = currentStack.pop();
                        assert !isWide(v1);
                        assert !isWide(v2);
                        assert !isWide(v3);
                        assert !isWide(v4);
                        currentStack.push(v2);
                        currentStack.push(v1);
                        currentStack.push(v4);
                        currentStack.push(v3);
                        currentStack.push(v2);
                        currentStack.push(v1);
                        break;
                    }
                } else {
                    // Form 2/4
                    if (isWide(v2)) {
                        // Form 4
                        // v2, v1 -> v1, v2, v1
                        assert isWide(v1);
                        assert isWide(v2);
                        currentStack.push(v1);
                        currentStack.push(v2);
                        currentStack.push(v1);
                        break;
                    } else {
                        LocalVariable v3 = currentStack.pop();
                        // Form 2
                        // v3, v2, v1 -> v1, v3, v2, v1
                        assert isWide(v1);
                        assert !isWide(v2);
                        assert !isWide(v3);
                        currentStack.push(v1);
                        currentStack.push(v3);
                        currentStack.push(v2);
                        currentStack.push(v1);
                        break;
                    }
                }
            }
            case SWAP: {
                LocalVariable v1 = currentStack.pop();
                LocalVariable v2 = currentStack.pop();
                currentStack.push(v1);
                currentStack.push(v2);
                break;
            }
            case IADD:
            case LADD:
            case FADD:
            case DADD:
            case ISUB:
            case LSUB:
            case FSUB:
            case DSUB:
            case IMUL:
            case LMUL:
            case FMUL:
            case DMUL:
            case IDIV:
            case LDIV:
            case FDIV:
            case DDIV:
            case IREM:
            case LREM:
            case FREM:
            case DREM:
            case ISHL:
            case LSHL:
            case ISHR:
            case LSHR:
            case IUSHR:
            case LUSHR:
            case IAND:
            case LAND:
            case IOR:
            case LOR:
            case IXOR:
            case LXOR: {
                Instruction right = pop();
                Instruction left = pop();
                Type result = NUMERIC_RESULTS[opcode - IADD];
                BinaryOp op = requireNonNull(NUMERIC_OPS[opcode - IADD]);

                assertStackType(left, result);
                assertStackType(right, result);

                addInsn(push(new Binary(op, left, right), result));
                break;
            }
            case INEG:
            case LNEG:
            case FNEG:
            case DNEG: { //Negations are converted to `0 - value`.
                Type result = NUMERIC_RESULTS[opcode - IADD];
                Instruction right = pop();
                Instruction left = NEG_LDC[opcode - INEG].copy();

                assertStackType(left, result);
                assertStackType(right, result);

                addInsn(push(new Binary(BinaryOp.SUB, left, right), result));
                break;
            }
            case I2L:
            case I2F:
            case I2D:
            case L2I:
            case L2F:
            case L2D:
            case F2I:
            case F2L:
            case F2D:
            case D2I:
            case D2L:
            case D2F:
            case I2B:
            case I2C:
            case I2S: {
                Type[] types = PRIMITIVE_CAST[opcode - I2L];
                Instruction value = pop();

                assertStackType(value, types[0]);

                addInsn(push(new Cast(value, typeResolver.resolveType(types[1])), types[1]));
                break;
            }
            case LCMP:    // left(1) > right(0) Long
            case FCMPL:   // left(1) > right(0) Float
            case FCMPG:   // left(1) < right(0) Float
            case DCMPL:   // left(1) > right(0) Double
            case DCMPG: { // left(1) < right(0) Double
                Type expectedType = COMPARISONS[opcode - LCMP];
                Instruction right = pop();
                Instruction left = pop();

                assertStackType(left, expectedType);
                assertStackType(right, expectedType);

                addInsn(push(new Compare(CMP_EXT_KINDS[opcode - LCMP], left, right), Type.INT_TYPE));
                break;
            }
            case IRETURN:
            case LRETURN:
            case FRETURN:
            case DRETURN:
            case ARETURN:
                vlGraph.markNode("Return");
                assert currentStack.size() == 1;
                Instruction arg = pop();
                addInsn(new Return(function, arg));
                break;
            case RETURN:
                vlGraph.markNode("Return");
                assert currentStack.isEmpty();
                addInsn(new Return(function));
                break;
            case ARRAYLENGTH:
                addInsn(push(new ArrayLen(pop()), Type.INT_TYPE));
                break;
            case ATHROW:
                vlGraph.markNode("Throw");
                addInsn(new Throw(pop()));
                break;
            case MONITORENTER:
                addInsn(Monitor.monitorEnter(pop()));
                break;
            case MONITOREXIT:
                addInsn(Monitor.monitorExit(pop()));
                break;
            default:
                throw new IllegalArgumentException("Unhandled insn: " + insnName);
        }
    }

    @Override
    public void visitIntInsn(int opcode, int operand) {
        switch (opcode) {
            case BIPUSH:
                addInsn(push(new LdcNumber(operand), Type.BYTE_TYPE));
                break;
            case SIPUSH:
                addInsn(push(new LdcNumber(operand), Type.SHORT_TYPE));
                break;
            case NEWARRAY:
                switch (operand) {
                    case T_BOOLEAN:
                    case T_CHAR:
                    case T_FLOAT:
                    case T_DOUBLE:
                    case T_BYTE:
                    case T_SHORT:
                    case T_INT:
                    case T_LONG: {
                        Instruction size = pop();
                        assertStackType(size, Type.INT_TYPE);
                        addInsn(push(new NewArray((ArrayType) typeResolver.resolveType(Type.getType("[" + T_TYPES[operand - T_BOOLEAN])), false, size), TypeResolver.OBJECT_TYPE));
                        break;
                    }
                    default:
                        throw new IllegalArgumentException("Unexpected NEWARRAY operand: " + operand);
                }
                break;
            default:
                throw new IllegalArgumentException("Unhandled int insn: " + OpcodeLookup.getName(opcode));
        }
    }

    @Override
    public void visitVarInsn(int opcode, int var) {
        switch (opcode) {
            case ALOAD: {
                if ((mNode.access & ACC_STATIC) == 0 && var == 0) {
                    addInsn(push(new LoadThis(thisType.get()), TypeResolver.OBJECT_TYPE));
                    break;
                }
            }
            case ILOAD:
            case LLOAD:
            case FLOAD:
            case DLOAD: {
                Type expectedType = ILFDA_TYPES[opcode - ILOAD];
                addInsn(push(new Load(getLoadLocal(var)), expectedType));
                break;
            }
            case ISTORE:
            case LSTORE:
            case FSTORE:
            case DSTORE:
            case ASTORE: {
                Instruction arg = pop();
                LocalVariable variable = getStoreLocal(var, arg.getResultType());
                addInsn(new Store(new LocalReference(variable), arg));
                break;
            }
            default:
                throw new IllegalArgumentException("Unhandled var insn: " + OpcodeLookup.getName(opcode));
            case RET:
                throw new IllegalArgumentException("Legacy RET instruction not currently supported.");
        }
    }

    @Override
    public void visitTypeInsn(int opcode, String typeDesc) {
        Type elementType = Type.getObjectType(typeDesc);
        AType type = typeResolver.resolveType(Type.getObjectType(typeDesc));
        switch (opcode) {
            case NEW:
                addInsn(push(new NewObject(type), TypeResolver.OBJECT_TYPE));
                break;
            case ANEWARRAY: {
                Instruction size = pop();
                assertStackType(size, Type.INT_TYPE);
                ArrayType arrayType = (ArrayType) typeResolver.resolveType(Type.getType("[" + elementType.getDescriptor()));
                addInsn(push(new NewArray(arrayType, false, size), TypeResolver.OBJECT_TYPE));
                break;
            }
            case CHECKCAST: {
                Instruction arg = pop();
                assertStackType(arg, TypeResolver.OBJECT_TYPE);

                // assert !TypeSystem.isAssignableTo(erasedArgType, type);
                // ^ fails because older javac sometimes inserts redundant casts. They get cleaned up later
                if (!TypeSystem.isCastableTo((ReferenceType) arg.getResultType(), (ReferenceType) type, true)) {
                    arg = new Cast(arg, typeResolver.resolveReferenceType(TypeResolver.OBJECT_TYPE));
                }
                addInsn(push(new Cast(arg, type), TypeResolver.OBJECT_TYPE));
                break;
            }
            case INSTANCEOF: {
                Instruction arg = pop();
                assertStackType(arg, TypeResolver.OBJECT_TYPE);
                addInsn(push(new InstanceOf(arg, type), Type.BOOLEAN_TYPE));
                break;
            }
            default:
                throw new IllegalArgumentException("Unhandled type insn: " + OpcodeLookup.getName(opcode));
        }
    }

    @Override
    public void visitFieldInsn(int opcode, String owner, String name, String descriptor) {
        ClassType clazz = typeResolver.resolveClassDecl(owner);
        Type desc = Type.getType(descriptor);
        Field field = requireNonNull(clazz.resolveField(name, desc), "Failed to resolve field " + owner + "." + name + " " + descriptor).asRaw();
        switch (opcode) {
            case GETSTATIC:
                addInsn(pushUnchecked(new Load(new FieldReference(clazz, field, new Nop()))));
                break;
            case PUTSTATIC: {
                Instruction value = pop();
                addInsn(new Store(new FieldReference(clazz, field, new Nop()), value));
                break;
            }
            case GETFIELD:
                addInsn(pushUnchecked(new Load(new FieldReference(clazz, field, pop()))));
                break;
            case PUTFIELD: {
                Instruction value = pop();
                Instruction target = pop();
                addInsn(new Store(new FieldReference(clazz, field, target), value));
                break;
            }
            default:
                throw new IllegalArgumentException("Unhandled field insn: " + OpcodeLookup.getName(opcode));
        }
    }

    @Override
    public void visitMethodInsn(int opcode, String ownerName, String name, String descriptor, boolean isInterface) {
        ReferenceType owner = (ReferenceType) typeResolver.resolveTypeDecl(Type.getObjectType(ownerName));
        Type methodDesc = Type.getMethodType(descriptor);
        Method method = requireNonNull(owner.resolveMethod(name, methodDesc), "Failed to resolve method " + ownerName + "." + name + descriptor).asRaw();
        ClassType targetClassType = owner instanceof ClassType ? (ClassType) owner : method.getDeclaringClass();
        List<Instruction> args = popMany(methodDesc.getArgumentTypes().length);
        switch (opcode) {
            case INVOKEVIRTUAL:
            case INVOKESPECIAL:
            case INVOKEINTERFACE: {
                Instruction target = pop();
                Instruction invoke = new Invoke(INVOKE_KINDS[opcode - INVOKEVIRTUAL], targetClassType, method, target, args);
                AType result = invoke.getResultType();
                Type retType = methodDesc.getReturnType();
                if (result != PrimitiveType.VOID && retType != Type.VOID_TYPE) {
                    // Polymorphic methods compile to whatever they are cast to in source.
                    // We need to recover the cast as the method will always return object.
                    if (method instanceof PolymorphicSignatureMethod && !retType.equals(TypeResolver.OBJECT_TYPE)) {
                        invoke = new Cast(invoke, typeResolver.resolveType(retType));
                    }
                    invoke = push(invoke, retType);
                }
                addInsn(invoke);
                break;
            }
            case INVOKESTATIC:
                Instruction invoke = new Invoke(INVOKE_KINDS[opcode - INVOKEVIRTUAL], targetClassType, method, new Nop(), args);
                if (invoke.getResultType() != PrimitiveType.VOID) {
                    invoke = push(invoke, methodDesc.getReturnType());
                }
                addInsn(invoke);
                break;
            default:
                throw new IllegalArgumentException("Unhandled method insn: " + OpcodeLookup.getName(opcode));
        }
    }

    @Override
    public void visitInvokeDynamicInsn(String name, String descriptor, Handle bootstrapMethodHandle, Object... bootstrapMethodArguments) {
        Type retType = Type.getReturnType(descriptor);
        InvokeDynamic invoke = new InvokeDynamic(
                typeResolver.resolveType(retType),
                name,
                (Method) parseHandle(bootstrapMethodHandle),
                FastStream.of(bootstrapMethodArguments)
                        .map(this::parseBootstrapArgument)
                        .toArray(),
                popMany(Type.getArgumentTypes(descriptor).length)
        );
        addInsn(push(invoke, retType));
    }

    private Object parseBootstrapArgument(Object obj) {
        if (obj instanceof ConstantDynamic) throw new NotImplementedException("ConstantDynamic not supported yet.");
        if (obj instanceof Handle) return parseHandle((Handle) obj);
        return obj;
    }

    private Object parseHandle(Handle handle) {
        ClassType clazz = typeResolver.resolveClassDecl(handle.getOwner());
        if (handle.getTag() <= H_PUTSTATIC) {
            return Objects.requireNonNull(clazz.resolveField(handle.getName(), Type.getType(handle.getDesc())), "Failed to resolve field handle." + handle).asRaw();
        }
        return Objects.requireNonNull(clazz.resolveMethod(handle.getName(), Type.getMethodType(handle.getDesc())), "Failed to resolve method handle. " + handle).asRaw();
    }

    @Override
    public void visitJumpInsn(int opcode, Label label) {
        Block target = getBlock(label);
        Instruction insn;
        switch (opcode) {
            case IFEQ:   // value == 0 jmp label
            case IFNE:   // value != 0 jmp label
            case IFLT:   // value <  0 jmp label
            case IFGE:   // value >= 0 jmp label
            case IFGT:   // value >  0 jmp label
            case IFLE: { // value <= 0 jmp label
                Comparison comparison = new Comparison(IF_CMP_KINDS[opcode - IFEQ], pop(), new LdcNumber(0));
                insn = new IfInstruction(comparison, new Branch(target));
                break;
            }
            case IF_ICMPEQ:   // value1(1) == value2(0) jmp label //int
            case IF_ICMPNE:   // value1(1) != value2(0) jmp label //int
            case IF_ICMPLT:   // value1(1) <  value2(0) jmp label //int
            case IF_ICMPGE:   // value1(1) >= value2(0) jmp label //int
            case IF_ICMPGT:   // value1(1) >  value2(0) jmp label //int
            case IF_ICMPLE:   // value1(1) <= value2(0) jmp label //int
            case IF_ACMPEQ:   // value1(1) == value2(0) jmp label //Object
            case IF_ACMPNE: { // value1(1) != value2(0) jmp label //Object
                Instruction right = pop();
                Instruction left = pop();
                Comparison comparison = new Comparison(IF_CMP_KINDS[opcode - IFEQ], left, right);
                insn = new IfInstruction(comparison, new Branch(target));
                break;
            }
            case GOTO:
                insn = new Branch(target);
                break;
            case IFNULL:      // value == null  jmp label
            case IFNONNULL: { // value != null  jmp label
                Comparison.ComparisonKind kind = opcode == IFNULL ? Comparison.ComparisonKind.EQUAL : Comparison.ComparisonKind.NOT_EQUAL;
                Comparison comparison = new Comparison(kind, pop(), new LdcNull());
                insn = new IfInstruction(comparison, new Branch(target));
                break;
            }
            default:
                throw new IllegalArgumentException("Unhandled int insn: " + OpcodeLookup.getName(opcode));
            case JSR://TODO JSRInlinerAdapter exists.
                throw new IllegalArgumentException("Legacy JSR instruction not currently supported.");
        }
        addInsn(insn);
    }

    @Override
    public void visitLabel(Label label) {
        activeLVNodes.removeIf(n -> n.end.getLabel() == label);
        if (importantLabels.containsKey(label)) {
            visitImportantLabel(label);
        }

        if (currentBlock == null) return;

        FastStream.of(mNode.localVariables).filter(n -> n.start.getLabel() == label).forEach(activeLVNodes::add);
        for (LocalVariableNode lv : activeLVNodes) {
            applyLVInfo(lv);
        }
    }

    public void visitImportantLabel(Label label) {
        Block block = getBlock(label);
        if (currentBlock != null) {
            addInsn(new Branch(block));
        }

        if (label == labelAfterCode || vlGraph.isDead(block)) {
            // we don't care about the label which is sometimes generated at the end of the method
            // this is the only time we should encounter a label with no incoming branches or fallthrough
            return;
        }

        assert currentStack.isEmpty();
        currentStack.addAll(vlGraph.visitBlock(block));

        FastStream.of(mNode.tryCatchBlocks)
                .filter(tcNode -> tcNode.end.getLabel() == label)
                .map(tryCatches::get)
                .distinct()
                .forEach(tc -> {
                    assert tc == currentContainer.getParent();
                    currentContainer = (BlockContainer) tc.getParent().getParent();
                });

        currentBlock = block;
        block.setBytecodeOffset(currentIndex);
        currentContainer.blocks.add(block);

        // Any try-catch blocks that start in this range the handler needs a link
        // back to the start of the try-catch range for variable resolution.
        FastStream.of(Lists.reverse(mNode.tryCatchBlocks))
                .filter(tcNode -> tcNode.start.getLabel() == label)
                .map(tryCatches::get)
                .distinct()
                .forEach(tryCatch -> {
                    vlGraph.addHandlerLink(getHandlerBlock(tryCatch));

                    if (tryCatch.getParentOrNull() == null) {
                        assert tryCatch.getTryBody().blocks.isEmpty();
                        currentBlock.instructions.add(tryCatch);
                        tryCatch.setBytecodeOffset(currentIndex);
                        currentBlock = new Block(currentBlock.getSubName("try")).withOffsets(currentBlock);
                    } else {
                        assert currentContainer == tryCatch.getParent().getParent();
                    }
                    currentContainer = tryCatch.getTryBody();
                    currentContainer.blocks.add(currentBlock);
                });
    }

    private void applyLVInfo(LocalVariableNode lv) {
        if (lv.index < firstLocalIndex) {
            return;
        }

        Type lvDesc = Type.getType(lv.desc);
        LocalVariable variable = new LocalVariable(
                VariableKind.LOCAL,
                typeResolver.resolveType(lvDesc),
                lv.signature,
                lv.index,
                lv.name
        );
        variable.setAnnotationSupplier(new AnnotationSupplier(typeResolver, FastStream.of(), FastStream.of(Util.safeConcat(mNode.visibleLocalVariableAnnotations, mNode.invisibleLocalVariableAnnotations))
                .filter(e -> e.index.contains(lv.index) && e.start.contains(lv.start) && e.end.contains(lv.end))
                .toList(FastStream.infer()))
        );
        vlGraph.applyLVInfo(variable);
    }

    @Override
    public void visitLdcInsn(Object value) {
        if (value instanceof Integer) {
            addInsn(push(new LdcNumber((Integer) value), Type.INT_TYPE));
        } else if (value instanceof Float) {
            addInsn(push(new LdcNumber((Float) value), Type.FLOAT_TYPE));
        } else if (value instanceof Long) {
            addInsn(push(new LdcNumber((Long) value), Type.LONG_TYPE));
        } else if (value instanceof Double) {
            addInsn(push(new LdcNumber((Double) value), Type.DOUBLE_TYPE));
        } else if (value instanceof String) {
            addInsn(push(new LdcString(typeResolver.resolveClass(TypeResolver.STRING_TYPE), (String) value), TypeResolver.OBJECT_TYPE));
        } else if (value instanceof Type) {
            Type type = (Type) value;
            int sort = type.getSort();
            if (sort == Type.OBJECT || sort == Type.ARRAY) {
                ReferenceType ldcType = (ReferenceType) typeResolver.resolveType(type);
                ClassType classClass = typeResolver.resolveClassDecl(TypeResolver.CLASS_TYPE);
                addInsn(push(new LdcClass(ldcType, new ParameterizedClass(null, classClass, ImmutableList.of(ldcType))), TypeResolver.OBJECT_TYPE));
            } else if (sort == Type.METHOD) {
                throw new UnsupportedOperationException("Not yet implemented.");
            } else {
                throw new IllegalArgumentException("Unhandled LDC Type " + sort);
            }
        } else if (value instanceof Handle) {
            throw new UnsupportedOperationException("Not yet implemented.");
        } else if (value instanceof ConstantDynamic) {
            throw new UnsupportedOperationException("Not yet implemented.");
        } else {
            throw new IllegalArgumentException("Unknown LDC object " + value.getClass().getName());
        }
    }

    @Override
    public void visitIincInsn(int var, int increment) {
        CompoundAssignment iinc = new CompoundAssignment(increment < 0 ? BinaryOp.SUB : BinaryOp.ADD, getLoadLocal(var), new LdcNumber(Math.abs(increment)));
        iinc.setTag(new IIncTag());
        addInsn(iinc);
    }

    @Override
    public void visitTableSwitchInsn(int min, int max, Label dflt, Label... labels) {
        assert max - min + 1 == labels.length;
        visitLookupSwitchInsn(dflt, IntStream.range(min, max + 1).toArray(), labels);
    }

    @Override
    public void visitLookupSwitchInsn(Label dflt, int[] keys, Label[] labels) {
        assert keys.length == labels.length;
        Instruction value = pop();
        assertStackType(value, Type.INT_TYPE);

        SwitchTable sw = new SwitchTable(value);

        Map<Label, List<Instruction>> keyMap = new HashMap<>();

        for (int i = 0; i < keys.length; i++) {
            if (labels[i] == dflt) continue;
            keyMap.computeIfAbsent(labels[i], e -> new LinkedList<>()).add(new LdcNumber(keys[i]));
        }
        keyMap.computeIfAbsent(dflt, e -> new LinkedList<>()).add(new Nop());
        FastStream.of(keyMap.entrySet())
                .sorted(Comparator.comparingInt(e -> importantLabels.getInt(e.getKey())))
                .forEach(e -> {
                    SwitchTable.SwitchSection section = new SwitchTable.SwitchSection(new Branch(getBlock(e.getKey())));
                    section.values.addAll(e.getValue());
                    sw.sections.add(section);
                });
        addInsn(sw);
    }

    @Override
    public void visitMultiANewArrayInsn(String descriptor, int numDimensions) {
        addInsn(push(new NewArray((ArrayType) typeResolver.resolveType(Type.getType(descriptor)), false, popMany(numDimensions)), TypeResolver.OBJECT_TYPE));
    }

    @Override
    public void visitFrame(int type, int numLocal, Object[] locals, int numStack, Object[] stacks) {
        assert currentStack.size() == numStack;
        // may be of use in the future for synthetic locals
        /*for (int i = firstLocalIndex; i < locals.length; i++) {
            Object local = locals[i];
            if (local instanceof String) {
                IType iType = typeResolver.resolveType(Type.getObjectType((String) local));
                InsnVariable v = Objects.requireNonNull(getReplacement(currentNode.readLocal(i)));
                makeVariableCompatibleWithType(iType, v);
            }
        }*/
    }

    @Override
    public void visitLineNumber(int line, Label start) {
        sourceLineNumber = line;
    }

    // THICC :D
    private boolean isWide(LocalVariable var) {
        AType declType = var.getType();
        return declType == PrimitiveType.DOUBLE || declType == PrimitiveType.LONG;
    }

    private void assertStackType(AType type, Type expectedStackType) {
        // boolean arrays BLOAD as Byte
        if (type == PrimitiveType.BOOLEAN && expectedStackType == Type.BYTE_TYPE || expectedStackType == Type.INT_TYPE) return;

        assert isAssignableTo(type, typeResolver.resolveType(expectedStackType));
    }

    private void assertStackType(Instruction insn, Type expectedStackType) {
        assertStackType(insn.getResultType(), expectedStackType);
    }

    private static Block getHandlerBlock(TryCatch tryCatch) {
        return ((Branch) tryCatch.handlers.only().getBody().getEntryPoint().instructions.only()).getTargetBlock();
    }

    @Nullable
    private static Label getFirstLabel(MethodNode mNode) {
        for (AbstractInsnNode instruction : mNode.instructions) {
            if (instruction.getType() == AbstractInsnNode.LABEL) {
                return ((LabelNode) instruction).getLabel();
            }
        }
        return null;
    }

    private static Object2IntMap<Label> computeImportantLabels(MethodNode mNode) {
        Object2IntMap<Label> labelNames = new Object2IntOpenHashMap<>();
        Set<Label> importantLabels = new HashSet<>();
        importantLabels.add(requireNonNull(getFirstLabel(mNode)));
        for (AbstractInsnNode insn : mNode.instructions) {
            switch (insn.getType()) {
                case AbstractInsnNode.LABEL:
                    LabelNode lNode = ((LabelNode) insn);
                    labelNames.put(lNode.getLabel(), labelNames.size());
                    break;
                case AbstractInsnNode.JUMP_INSN:
                    importantLabels.add(((JumpInsnNode) insn).label.getLabel());
                    break;
                case AbstractInsnNode.TABLESWITCH_INSN:
                    TableSwitchInsnNode tsinsn = (TableSwitchInsnNode) insn;
                    importantLabels.add(tsinsn.dflt.getLabel());
                    tsinsn.labels.forEach(e -> importantLabels.add(e.getLabel()));
                    break;
                case AbstractInsnNode.LOOKUPSWITCH_INSN:
                    LookupSwitchInsnNode lsinsn = (LookupSwitchInsnNode) insn;
                    importantLabels.add(lsinsn.dflt.getLabel());
                    lsinsn.labels.forEach(e -> importantLabels.add(e.getLabel()));
                    break;
            }
        }
        for (TryCatchBlockNode tryCatchBlock : mNode.tryCatchBlocks) {
            importantLabels.add(tryCatchBlock.start.getLabel());
            importantLabels.add(tryCatchBlock.end.getLabel());
            importantLabels.add(tryCatchBlock.handler.getLabel());
        }
        Object2IntMap<Label> ret = new Object2IntOpenHashMap<>();
        for (Label label : importantLabels) {
            assert labelNames.containsKey(label);
            ret.put(label, labelNames.getInt(label));
        }
        return ret;
    }
}
