package net.covers1624.coffeegrinder.bytecode.flow;

import net.covers1624.coffeegrinder.bytecode.SimpleInsnVisitor;
import net.covers1624.coffeegrinder.bytecode.insns.Block;
import net.covers1624.coffeegrinder.bytecode.insns.BlockContainer;
import net.covers1624.coffeegrinder.bytecode.insns.Branch;
import net.covers1624.coffeegrinder.bytecode.insns.Leave;
import net.covers1624.coffeegrinder.util.None;

import java.util.*;

/**
 * Holds the control flow graph.
 * A separate graph is computed for each {@link BlockContainer} at the start of block transforms
 * (before loop detection)
 * <p>
 * Created by covers1624 on 19/4/21.
 */
public class ControlFlowGraph {

    /**
     * The container for which the {@link ControlFlowGraph} was created.
     * <p>
     * This may differ from the container currently holding a block,
     * because a transform could have moved the block since the CFG was created.
     */
    public final BlockContainer container;

    /**
     * Nodes array, indexed by original block index.
     * <p>
     * Originally <code>cfg[i].block == container.blocks.get(i)</code>
     * but may have since been moved/reordered by transforms.
     */
    public final ControlFlowNode[] cfg;

    /**
     * A map of {@link Block} to {@link ControlFlowNode}.
     * <p>
     * Unlike the cfg array, this can be used to discover control flow nodes even after
     * blocks were moved/reordered by transforms.
     */
    public final Map<Block, ControlFlowNode> nodes = new HashMap<>();

    /**
     * <code>directExits.get(i) == true</code> if <code>cfg[i]</code> directly contains a
     * branch/leave instruction leaving {@link #container}.
     */
    public final BitSet directExits;

    /**
     * <code>reachableExits.get(i) == true</code> if there is a path from <code>cfg[i]</code>
     * to a node not dominated by <code>cfg[i]</code>, or if there is a path from <code>cfg[i]</code>
     * to a branch/leave instruction leaving the {@link #container}.
     */
    public final BitSet reachableExits;

    /**
     * Constructs a control flow graph for the blocks in the given block container.
     * <p>
     * Return statements, exceptions, or branches leaving the block container are not
     * modeled by the control flow graph.
     *
     * @param container The container.
     */
    public ControlFlowGraph(BlockContainer container) {
        this.container = container;
        List<Block> blocks = container.blocks.toList();
        cfg = new ControlFlowNode[blocks.size()];
        directExits = new BitSet(cfg.length);
        reachableExits = new BitSet(cfg.length);

        for (int i = 0; i < cfg.length; i++) {
            Block block = blocks.get(i);
            cfg[i] = new ControlFlowNode(i, block);
            nodes.put(block, cfg[i]);
        }
        createEdges(blocks);
        Dominance.computeDominance(cfg[0]);
        Dominance.computeReachableExits(reachableExits, cfg);
        computeExits();
    }

    private void createEdges(List<Block> blocks) {
        for (int i = 0; i < blocks.size(); i++) {
            Block block = blocks.get(i);
            ControlFlowNode sourceNode = cfg[i];

            // todo, could be faster by following the incoming branches of blocks, and leaves of container and parents
            int finalI = i;
            block.accept(new SimpleInsnVisitor<None>() {
                @Override
                public None visitBranch(Branch branch, None ctx) {
                    if (branch.getTargetBlock().getParentOrNull() == container) {
                        sourceNode.addEdgeTo(nodes.get(branch.getTargetBlock()));
                    } else if (!branch.getTargetBlock().isDescendantOf(container)) {
                        // Branch out of this container into a parent container.
                        // Like return statements and exceptional exits,
                        // we ignore this for the CFG and the dominance calculation.
                        // However, it's relevant for hasReachableExit().
                        directExits.set(finalI);
                    }
                    // else: Internal control flow within a nested container.
                    return super.visitBranch(branch, ctx);
                }

                @Override
                public None visitLeave(Leave leave, None ctx) {
                    if (!leave.getTargetContainer().isDescendantOf(block)) {
                        // Leave instructions (like other exits out of the container)
                        // are ignored for the CFG and dominance,
                        // but is relevant for hasReachableExit().
                        directExits.set(finalI);
                    }
                    return super.visitLeave(leave, ctx);
                }
            });
        }
    }

    private void computeExits() {
        // Also mark the nodes that exit the block container altogether.
        // Invariant: leaving.get(n.cfgIndex) == true implies leaving.get(n.getImmediateDominator().cfgIndex) == true
        BitSet leaving = new BitSet(cfg.length);
        for (ControlFlowNode node : cfg) {
            if (leaving.get(node.cfgIndex)) continue;

            if (directExits.get(node.cfgIndex)) {
                for (ControlFlowNode p = node; p != null; p = p.immediateDominator) {
                    if (leaving.get(p.cfgIndex)) {
                        // we can stop marking when we've reached an already-marked nodes
                        break;
                    }
                    leaving.set(p.cfgIndex);
                }
            }
        }
        reachableExits.or(leaving);
    }

    /**
     * Gets the control flow node for the block.
     * <p>
     * Precondition: the block belonged to the {@link #container} at the start of the block transforms
     * (where the control flow graph was created).
     *
     * @param block The block to get the {@link ControlFlowNode} for.
     * @return The ControlFlowNode.
     */
    public ControlFlowNode getNode(Block block) {
        return Objects.requireNonNull(nodes.get(block));
    }

    /**
     * Returns true if there is a control flow path from <code>node</code> to one of the following:
     * - Branch or leave instruction leaving <code>this.container</code>.
     * - Branch instruction within this container to another node that is not dominated by <code>node</code>.
     * <p>
     * If this function returns <code>false</code>, the only way control flow can leave the set of nodes
     * dominated by <code>node</code> is by executing a <code>return;</code> or <code>throw;</code> instruction.
     *
     * @param node The node to check.
     * @return If the node has a reachable exit.
     */
    public boolean hasReachableExit(ControlFlowNode node) {
        assert cfg[node.cfgIndex] == node;
        return reachableExits.get(node.cfgIndex);
    }

    /**
     * Gets whether the control flow node directly contains a branch/leave instruction
     * exiting the container.
     *
     * @param node The node to check.
     * @return If the node has a direct exit.
     */
    public boolean hasDirectExit(ControlFlowNode node) {
        assert cfg[node.cfgIndex] == node;
        return directExits.get(node.cfgIndex);
    }

    /**
     * Resets all {@link ControlFlowNode#visited} flags for all nodes in this graph.
     */
    public void resetVisited() {
        for (ControlFlowNode node : cfg) {
            node.visited = false;
        }
    }

}
