package net.covers1624.coffeegrinder.bytecode.insns;

import net.covers1624.coffeegrinder.bytecode.*;
import net.covers1624.coffeegrinder.type.AType;
import net.covers1624.coffeegrinder.type.PrimitiveType;
import net.covers1624.coffeegrinder.util.EnumBitSet;
import net.covers1624.quack.collection.FastStream;

import java.util.LinkedList;
import java.util.List;

/**
 * Created by covers1624 on 22/12/21.
 */
public class Switch extends Instruction {

    private final InstructionSlot<Instruction> value = new InstructionSlot<>(this);
    private final InstructionSlot<BlockContainer> body = new InstructionSlot<>(this);

    private final List<Yield> yields = new LinkedList<>();
    private final AType resultType;

    public Switch(Instruction value, BlockContainer body) {
        this(value, body, PrimitiveType.VOID);
    }

    public Switch(Instruction value, BlockContainer body, AType resultType) {
        super(InsnOpcode.SWITCH);
        this.resultType = resultType;
        this.value.set(value);
        this.body.set(body);
    }

    @Override
    public AType getResultType() {
        return resultType;
    }

    @Override
    public EnumBitSet<InstructionFlag> getDirectFlags() {
        return InstructionFlag.NONE;
    }

    @Override
    protected EnumBitSet<InstructionFlag> computeFlags() {
        // Switch expressions don't propagate any flags out.
        if (resultType != PrimitiveType.VOID) return InstructionFlag.NONE;

        EnumBitSet<InstructionFlag> flags = super.computeFlags();
        if (!getSwitchTable().isExhaustive()) {
            flags.clear(InstructionFlag.END_POINT_UNREACHABLE);
        }
        return flags;
    }

    @Override
    public <R, C> R accept(InsnVisitor<R, C> visitor, C ctx) {
        return visitor.visitSwitch(this, ctx);
    }

    public Instruction getValue() {
        return value.get();
    }

    public BlockContainer getBody() {
        return body.get();
    }

    public SwitchTable getSwitchTable() {
        return (SwitchTable) getBody().getFirstChild().getFirstChild();
    }

    void addYield(Yield yield) {
        yields.add(yield);
    }

    void remYield(Yield yield) {
        yields.remove(yield);
    }

    public int getYieldCount() {
        return yields.size();
    }

    public FastStream<Yield> getYields() {
        return FastStream.of(yields);
    }
}
