package net.covers1624.coffeegrinder.bytecode.transform.transformers;

import net.covers1624.coffeegrinder.bytecode.InsnOpcode;
import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.SemanticMatcher;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.matching.InvokeMatching;
import net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching;
import net.covers1624.coffeegrinder.bytecode.transform.ClassTransformContext;
import net.covers1624.coffeegrinder.bytecode.transform.ClassTransformer;
import net.covers1624.coffeegrinder.type.Field;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.function.Supplier;

/**
 * Responsible for inlining Lambda synthetic methods.
 * <p>
 * Created by covers1624 on 7/8/21.
 */
public class FieldInitializers implements ClassTransformer {

    @SuppressWarnings ("NotNullFieldNotInitialized")
    private ClassTransformContext ctx;

    @Override
    public void transform(ClassDecl cInsn, ClassTransformContext ctx) {
        this.ctx = ctx;

        List<FieldDecl> staticFields = cInsn.getFieldMembers()
                .filter(f -> f.getField().isStatic())
                .toLinkedList();
        MethodDecl staticInitializer = cInsn.getMethodMembers().filter(f -> f.getMethod().getName().equals("<clinit>")).onlyOrDefault();
        if (staticInitializer != null) {
            Block staticInitializerBody = staticInitializer.getBody().getEntryPoint();
            processInitializers(staticFields, Collections.singletonList(staticInitializerBody.instructions::firstOrDefault));
        }

        List<FieldDecl> instanceFields = cInsn.getFieldMembers()
                .filter(f -> !f.getField().isStatic())
                .toLinkedList();
        processInitializers(instanceFields,
                cInsn.getMethodMembers()
                        .map(InvokeMatching::getSuperConstructorCall)
                        .filter(Objects::nonNull)
                        .<Supplier<@Nullable Instruction>>map(e -> e::getNextSiblingOrNull)
                        .toLinkedList()
        );
    }

    private void processInitializers(List<FieldDecl> fields, List<Supplier<@Nullable Instruction>> potentialFieldInitSuppliers) {
        // Proguard member-wise stripping may strip constructors for classes when they are unused.
        if (potentialFieldInitSuppliers.isEmpty()) return;

        int lastFieldIndex = -1;
        while (true) {
            Store stfld = LoadStoreMatching.matchStoreField(potentialFieldInitSuppliers.get(0).get());
            if (stfld == null) return;

            FieldDecl field = matchField(fields, ((FieldReference) stfld.getReference()).getField());
            if (field == null || fields.indexOf(field) <= lastFieldIndex) return;

            // make sure the value doesn't reference the constructor parameters
            Instruction value = stfld.getValue();
            if (!value.descendantsWhere(FieldInitializers::isLoadParameter).isEmpty()) return;

            List<Store> stflds = new LinkedList<>();
            stflds.add(stfld);

            for (int i = 1; i < potentialFieldInitSuppliers.size(); i++) {
                Instruction insn2 = potentialFieldInitSuppliers.get(i).get();
                if (insn2 == null || !new SemanticMatcher(null).equivalent(stfld, insn2)) {
                    return;
                }
                stflds.add((Store) insn2);
            }

            // Disallow multi-store being inlined.
            if (value.opcode == InsnOpcode.STORE) {
                return;
            }

            ctx.pushStep("Initializer for " + field.getField().getName());
            field.setValue(value);
            stflds.forEach(Instruction::remove);
            if (!field.getField().isSynthetic()) { // initializers for synthetic fields get inserted in different orders to the fields themselves.
                lastFieldIndex = fields.indexOf(field);
            }
            ctx.popStep();
        }
    }

    private static boolean isLoadParameter(Instruction instruction) {
        Load load = LoadStoreMatching.matchLoadLocal(instruction);
        return load != null && load.getVariable().getKind() == LocalVariable.VariableKind.PARAMETER;
    }

    @Nullable
    private FieldDecl matchField(List<FieldDecl> fields, Field field) {
        return FastStream.of(fields).filter(f -> f.getField().getName().equals(field.getName())).onlyOrDefault();
    }
}
