package net.covers1624.coffeegrinder.util.resolver;

import com.google.common.collect.ImmutableSet;
import net.covers1624.quack.collection.FastStream;
import net.covers1624.quack.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Set;
import java.util.stream.Stream;

/**
 * Created by covers1624 on 8/4/21.
 */
public abstract class AbstractPathResolver implements Resolver {

    protected final Path path;

    protected @Nullable Set<String> files;

    protected AbstractPathResolver(Path path) {
        this.path = path;
        validate();
    }

    //Validate this Resolver.
    protected abstract void validate();

    //Get the root path. Zip resolvers override this.
    protected Path getRootPath() {
        return path;
    }

    protected final Set<String> getFiles() {
        if (files == null) {
            synchronized (this) {
                if (files != null) return files;
                ImmutableSet.Builder<String> builder = ImmutableSet.builder();
                collectFiles(getRootPath(), builder);
                files = builder.build();
            }
        }
        return files;
    }

    private void collectFiles(Path path, ImmutableSet.Builder<String> files) {
        if (Files.isDirectory(path)) {
            try (Stream<Path> stream = Files.list(path)) {
                stream.forEach(e -> collectFiles(e, files));
            } catch (IOException ex) {
                throw new RuntimeException("Failed to list directory.", ex);
            }
        } else {
            // Replace all backslashes with forward slashes and remove leading '/'.
            files.add(StringUtils.stripStart(getRootPath().relativize(path).toString().replace("\\", "/"), "/"));
        }
    }

    @Override
    public byte @Nullable [] getResource(String name) throws IOException {
        if (!hasResource(name)) return null;

        Path path = getRootPath().resolve(name);
        try (InputStream is = Files.newInputStream(path)) {
            return IOUtils.toBytes(is);
        } catch (IOException ex) {
            throw new IOException("Failed to read resource at " + path + " with " + this, ex);
        }
    }

    @Override
    public boolean hasResource(String name) {
        return getFiles().contains(name);
    }

    @Override
    public FastStream<String> getAllClasses() {
        return FastStream.of(getFiles())
                .filter(e -> e.endsWith(".class"))
                .map(e -> e.replace(".class", ""));
    }

    @Override
    public FastStream<String> getAllResources() {
        return FastStream.of(getFiles())
                .filterNot(e -> e.endsWith(".class"));
    }
}
