package net.covers1624.coffeegrinder.bytecode.transform.transformers;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import net.covers1624.coffeegrinder.bytecode.InsnOpcode;
import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.SimpleInsnVisitor;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.transform.MethodTransformContext;
import net.covers1624.coffeegrinder.bytecode.transform.MethodTransformer;
import net.covers1624.coffeegrinder.util.None;
import net.covers1624.quack.collection.FastStream;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;

import static com.google.common.collect.ImmutableList.copyOf;
import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.matchLocalRef;
import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.matchStoreLocal;

/**
 * A variable is in scope, if the parent of the declaration (IStoreInstruction) is an ancestor of the usage
 * and the declaration has a lower child index in the common ancestor than the usage
 * <p>
 * Split variables with an aliased declaration in-scope are merged.
 * Created by covers1624 on 15/9/21.
 */
public class VariableDeclarations extends SimpleInsnVisitor<None> implements MethodTransformer {

    private final Map<LocalVariable, Instruction> varDeclPoints = new LinkedHashMap<>();

    @Override
    public void transform(MethodDecl function, MethodTransformContext ctx) {
        varDeclPoints.clear();
        function.accept(this);

        Multimap<String, Pair<Instruction, LocalVariable>> aliasedDeclarations = HashMultimap.create();

        for (Map.Entry<LocalVariable, Instruction> entry : varDeclPoints.entrySet()) {
            LocalVariable var = entry.getKey();
            Instruction declPoint = entry.getValue();

            LocalVariable aliasInScope = getDeclarationInScopeIfAny(aliasedDeclarations.get(var.getName()), declPoint);
            if (aliasInScope != null && replaceVariable(var, aliasInScope, ctx)) {
                continue;
            }

            if (!isDeclaration(declPoint, var)) {// is this insn a reference??
                ctx.pushStep("Declare " + var.getUniqueName() + " in outer scope");
                LocalReference newDecl = new LocalReference(var);
                declPoint.insertBefore(newDecl);
                declPoint = newDecl;
                ctx.popStep();
            }
            aliasedDeclarations.put(var.getName(), Pair.of(declPoint, var));
        }
    }

    private boolean isDeclaration(Instruction declPoint, LocalVariable var) {
        return matchLocalRef(declPoint, var) != null || matchStoreLocal(declPoint, var) != null;
    }

    @Nullable
    private LocalVariable getDeclarationInScopeIfAny(Collection<Pair<Instruction, LocalVariable>> aliasedDeclarations, Instruction usage) {
        return FastStream.of(aliasedDeclarations)
                .filter(pair -> usage.isDescendantOf(pair.getLeft().getParent()))
                .map(Pair::getRight)
                .onlyOrDefault();
    }

    private boolean replaceVariable(LocalVariable var, LocalVariable replacement, MethodTransformContext ctx) {
        // when decompiling without a LV table, we may want to see if we can merge the types of the variable. But that does require checking compliance with all usages...
        if (!var.getType().equals(replacement.getType()) || var.getIndex() != replacement.getIndex()) return false;
        ctx.pushStep("Merge variable " + var.getUniqueName() + " into " + replacement.getUniqueName());
        copyOf(var.getReferences()).forEach(r -> r.replaceWith(new LocalReference(replacement)));
        ctx.popStep();
        return true;
    }

    /**
     * Finds a suitable declaration location such that usage is in scope with the previous declaration.
     * Assumes that decl is 'before' usage
     *
     * @param decl  The previous declaration for a variable
     * @param usage An instruction which requires the declaration to be in scope
     * @return <code>decl</code>, if <code>decl</code> is in scope, otherwise a common ancestor of both <code>decl</code> and <code>usage</code>,
     * which is a child of a block (so a declaration could be inserted before it)
     */
    public static Instruction unifyUsages(Instruction decl, Instruction usage) {
        Instruction commonParent = findCommonParent(decl, usage);

        if (isDeclInInitializerSlot(decl, commonParent)) {
            return decl;
        }

        if (commonParent.opcode == InsnOpcode.BLOCK) { // select the current declaration within the common parent block
            return findAncestorChild(decl, commonParent);
        }

        // common parent may be a single instruction (eg an if)
        return selectDeclarableParent(commonParent);
    }

    private static boolean isDeclInInitializerSlot(Instruction decl, Instruction commonParent) {
        switch (commonParent.opcode) {
            case FOR_LOOP:
            case TRY_WITH_RESOURCES:
            case FOR_EACH_LOOP:
            case TRY_CATCH_HANDLER:
                return commonParent.getFirstChild() == decl;
            default:
                return false;
        }
    }

    /**
     * find the first declaration point, the first ancestor which is a child of a block (which may be this instruction, or a parent)
     */
    public static Instruction selectDeclarableParent(Instruction insn) {
        while (insn.getParent().opcode != InsnOpcode.BLOCK) {
            insn = insn.getParent();

            // an escape hatch for when the root container has multiple blocks. Indicates previous transforms have failed to remove all gotos
            if (insn.opcode == InsnOpcode.METHOD_DECL) {
                // declare at the top of the function. Hope for the best
                return ((MethodDecl) insn).getBody().getEntryPoint().getFirstChild();
            }
        }
        return insn;
    }

    private static Instruction findAncestorChild(Instruction insn, Instruction ancestor) {
        while (insn.getParent() != ancestor) {
            insn = insn.getParent();
        }

        return insn;
    }

    private static Instruction findCommonParent(Instruction decl, Instruction use2) {
        Instruction parent = decl;
        do {
            parent = parent.getParent();
        }
        while (!use2.isDescendantOf(parent));
        return parent;
    }

    @Override
    public None visitLocalReference(LocalReference localRef, None ctx) {
        LocalVariable var = localRef.variable;
        if (var.getKind() == LocalVariable.VariableKind.PARAMETER) return NONE;

        Instruction scope = localRef.getParent().opcode == InsnOpcode.STORE ? localRef.getParent() : localRef;
        Instruction decl = varDeclPoints.get(var);
        if (decl == null) {
            assert localRef.isWrittenTo();
            varDeclPoints.put(var, scope);
        } else {
            varDeclPoints.put(var, unifyUsages(decl, scope));
        }

        return NONE;
    }
}
