package net.covers1624.coffeegrinder.bytecode.flow;

import net.covers1624.coffeegrinder.bytecode.insns.Block;
import net.covers1624.coffeegrinder.debug.Debugger;
import net.covers1624.quack.collection.FastStream;
import net.covers1624.quack.util.SneakyUtils;
import org.jetbrains.annotations.Nullable;

import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Function;

/**
 * Represents a block in the control flow graph.
 * <p>
 * Created by covers1624 on 28/2/21.
 */
public class ControlFlowNode {

    /**
     * The index in this nodes control flow graph.
     */
    public int cfgIndex;

    /**
     * The block this control flow node is for.
     */
    @Nullable
    public Block block;

    /**
     * Visited flag, used in various algorithms.
     */
    public boolean visited;

    /**
     * The index of the node in a post-order traversal of the control flow graph, starting at the
     * end point. This field gets computed by dominance analysis.
     */
    public int postOrderNumber;

    /**
     * List of incoming control flow edges.
     */
    private final List<ControlFlowNode> predecessors = new ArrayList<>();

    /**
     * List of outgoing control flow edges.
     */
    private final List<ControlFlowNode> successors = new ArrayList<>();

    /**
     * The immediate dominator (the parent in the dominator tree).
     * <code>null</code> if the dominance has not been calculated or the node is unreachable.
     */
    @Nullable
    ControlFlowNode immediateDominator;

    /**
     * List of children in the dominator tree.
     * <code>null</code> if the dominance has not been calculated or the node is unreachable.
     */
    @Nullable
    List<ControlFlowNode> dominatorTreeChildren = null;

    public ControlFlowNode() { }

    public ControlFlowNode(int cfgIndex) {
        this.cfgIndex = cfgIndex;
    }

    public ControlFlowNode(ControlFlowNode other) {
        this.cfgIndex = other.cfgIndex;
        this.block = other.block;
    }

    public ControlFlowNode(int cfgIndex, Block block) {
        this.cfgIndex = cfgIndex;
        this.block = block;
    }

    public void addEdgeTo(ControlFlowNode target) {
        successors.add(target);
        target.predecessors.add(this);
    }

    public void traversePreOrder(Function<ControlFlowNode, List<ControlFlowNode>> children, Consumer<ControlFlowNode> visitor) {
        if (visited) return;
        visited = true;
        visitor.accept(this);
        for (ControlFlowNode t : children.apply(this)) {
            t.traversePreOrder(children, visitor);
        }
    }

    public void traversePostOrder(Function<ControlFlowNode, List<ControlFlowNode>> children, Consumer<ControlFlowNode> visitor) {
        if (visited) return;
        visited = true;
        for (ControlFlowNode t : children.apply(this)) {
            t.traversePostOrder(children, visitor);
        }
        visitor.accept(this);
    }

    /**
     * Gets whether this node dominates the provided node.
     *
     * @param node The node to check against.
     * @return If this node dominates the provided node.
     */
    public boolean dominates(ControlFlowNode node) {
        ControlFlowNode tmp = node;
        while (tmp != null) {
            if (tmp == this) return true;

            tmp = tmp.immediateDominator;
        }
        return false;
    }

    public FastStream<ControlFlowNode> streamPreOrder(Function<ControlFlowNode, FastStream<ControlFlowNode>> children) {
        return FastStream.of(this).concat(children.apply(this).flatMap(e -> e.streamPreOrder(children)));
    }

    public FastStream<ControlFlowNode> streamPostOrder(Function<ControlFlowNode, FastStream<ControlFlowNode>> children) {
        return children.apply(this).flatMap(e -> e.streamPostOrder(children)).concat(FastStream.of(this));
    }

    /**
     * Gets the {@link #block}, requiring it to be non-null.
     * (mostly for nullability convenience.)
     *
     * @return The block.
     */
    public Block getBlock() {
        return Objects.requireNonNull(block);
    }

    /**
     * @return List of incoming control flow edges.
     */
    public List<ControlFlowNode> getPredecessors() {
        return predecessors;
    }

    /**
     * @return List of outgoing control flow edges.
     */
    public List<ControlFlowNode> getSuccessors() {
        return successors;
    }

    /**
     * Gets the immediate dominator (the parent in the dominator tree).
     * <code>null</code> if the dominance has not been calculated; or the node is unreachable.
     * <p>
     * This is an overload for {@link #getImmediateDominatorOrNull()}
     *
     * @return The immediate parent, or null.
     */
    public ControlFlowNode getImmediateDominator() {
        return Objects.requireNonNull(getImmediateDominatorOrNull());
    }

    /**
     * Gets the immediate dominator (the parent in the dominator tree).
     * <code>null</code> if the dominance has not been calculated; or the node is unreachable.
     *
     * @return The immediate parent, or null.
     */
    @Nullable
    public ControlFlowNode getImmediateDominatorOrNull() {
        return immediateDominator;
    }

    /**
     * List of children in the dominator tree.
     *
     * @return The children.
     */
    public List<ControlFlowNode> getDominatorTreeChildren() {
        return Objects.requireNonNull(dominatorTreeChildren);
    }

    /**
     * Gets whether this node is reachable. Requires that dominance is computed!
     *
     * @return If the node is reachable.
     */
    public boolean isReachable() {
        return dominatorTreeChildren != null;
    }

    @Override
    public String toString() {
        if (block == null) return "Block at: UNKNOWN";
        return "Block at: " + block.getName();
    }

    // Exists for debugger evaluation.
    private void evalDumpGraph() {
        evalDumpGraph("eval_graph");
    }

    private void evalDumpGraph(String name) {
        Debugger.tryIfPresent(e -> e.writeControlFlowGraph(this, Paths.get("./graphs/" + name + ".png")));
    }
}
