package net.covers1624.coffeegrinder.type;

import net.covers1624.coffeegrinder.bytecode.AccessFlag;
import net.covers1624.coffeegrinder.util.EnumBitSet;
import net.covers1624.coffeegrinder.util.Util;
import net.covers1624.quack.collection.ColUtils;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;
import org.objectweb.asm.Type;

import java.util.*;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;
import static net.covers1624.coffeegrinder.type.TypeSubstitutions.applyOwnerParameterization;
import static net.covers1624.coffeegrinder.type.TypeSubstitutions.subst;
import static net.covers1624.quack.collection.FastStream.of;

/**
 * Created by covers1624 on 22/12/21.
 */
public final class ParameterizedClass extends ClassType implements TypeSubstitutions.TypeParamMapper {

    @Nullable
    private final ParameterizedClass outer;

    private final ClassType unboundClass;
    private final List<ReferenceType> arguments;

    private final Supplier<ClassType> superClass;
    private final Supplier<List<ClassType>> interfaces;

    private final Supplier<List<Field>> fields;
    private final Supplier<List<Method>> methods;

    public ParameterizedClass(@Nullable ParameterizedClass outer, ClassType unboundClass, List<ReferenceType> arguments) {
        this.outer = outer;
        assert unboundClass.getTypeParameters().size() == arguments.size() || arguments.isEmpty();
        assert unboundClass.getDeclaration() == unboundClass;
        assert (outer != null) == TypeSystem.needsOuterParameterization(unboundClass);

        this.unboundClass = unboundClass;
        this.arguments = arguments;
        superClass = Util.singleMemoize(() -> subst(unboundClass.getSuperClass(), this));
        interfaces = Util.singleMemoize(() -> of(unboundClass.getInterfaces()).map(e -> subst(e, this)).toImmutableList());

        fields = Util.singleMemoize(() -> of(unboundClass.getFields()).map(e -> applyOwnerParameterization(this, e)).toImmutableList());
        methods = Util.singleMemoize(() -> of(unboundClass.getMethods()).map(e -> applyOwnerParameterization(this, e)).toImmutableList());
    }

    @Override
    public ReferenceType mapParam(TypeParameter param) {
        if (param.getOwner() == unboundClass) {
            if (!arguments.isEmpty()) {
                return arguments.get(param.getIndex());
            }

            // Technically, we should ask outer to bind into the upper bound here as well, but javac doesn't support it :)
            return param;
        }

        return outer != null ? outer.mapParam(param) : param;
    }

    public ParameterizedClass capture() {
        ParameterizedClass outerCaptured = outer != null ? outer.capture() : null;

        if (outerCaptured == outer && !ColUtils.anyMatch(arguments, e -> e instanceof WildcardType)) return this;

        Map<ParameterizedClass, ParameterizedClass> derivedCaptures = new HashMap<>();
        // This class is not threadsafe, it is assumed that captures are local to an evaluation tree
        class Capture extends CapturedTypeVar {

            private ReferenceType upper;

            public Capture(TypeParameter orig, WildcardType wildcard) {
                super(orig, wildcard);
                upper = TypeSystem.objectType(orig); // recursion bailout, object is not assignable to anything except ?
            }

            @Override
            public ReferenceType getUpperBound() {
                return upper;
            }

            @Override
            public Capture map(TypeSubstitutions.TypeMapper map) {
                // TODO, probably some more bailouts for captures which obviously can't be affected by a remapper...
                //   just difficult because the applied argument for the upper bound could be mapped...

                ParameterizedClass mapped = (ParameterizedClass) TypeSubstitutions.subst(ParameterizedClass.this, map);
                if (mapped.equals(ParameterizedClass.this)) { return this; }

                return (Capture) derivedCaptures.computeIfAbsent(mapped, ParameterizedClass::capture).arguments.get(orig.getIndex());
            }

            @Override
            public boolean mentions(ReferenceType type) {
                if (super.mentions(type)) return true;

                // one interpretation of mentions could be `upper.mentions(type) || getLowerBound().mentions(type)`
                // however, that requires a recursion bailout, and also means that solving inference struggles:
                //   when recapturing (see map above), a capture which doesn't actually have any dependencies, but gets a new identity due to other args in the class
                //
                // this solution is non-recursive, and effectively, a capture of an argument to a parameterized class is effectively dependent on the whole set of arguments
                //   since a change to any of them could cause it to be remapped
                return orig.mentions(type) || ParameterizedClass.this.mentions(type);
            }
        }

        List<ReferenceType> fresh = new ArrayList<>(arguments.size());
        for (int i = 0; i < arguments.size(); i++) {
            ReferenceType typeArg = arguments.get(i);
            if (typeArg instanceof WildcardType) {
                fresh.add(new Capture(unboundClass.getTypeParameters().get(i), (WildcardType) typeArg));
            } else {
                fresh.add(typeArg);
            }
        }

        ParameterizedClass capture = new ParameterizedClass(
                outerCaptured,
                unboundClass,
                fresh);

        for (ReferenceType freshTypeArg : fresh) {
            if (!(freshTypeArg instanceof Capture)) continue;

            Capture c = (Capture) freshTypeArg;
            c.upper = TypeSystem.glbJavac(c.wildcard.getUpperBound(), subst(c.orig.getUpperBound(), capture));
            c.upper = TypeSystem.capture(c.upper);

            // class C<T, U extends T>
            // C<X, ? extends Y>
            // C<X, capture ? extends glb(X, Y)>
        }

        return capture;
    }

    //@formatter:off
    public List<ReferenceType> getTypeArguments() { return arguments; }
    @Override public ClassType getSuperClass() { return superClass.get(); }
    @Override public List<ClassType> getInterfaces() { return interfaces.get(); }
    @Override public List<ClassType> getNestedClasses() { throw new UnsupportedOperationException(); }
    @Override public List<Field> getFields() { return fields.get(); }
    @Override public List<Method> getMethods() { return methods.get(); }
    @Override public DeclType getDeclType() { return unboundClass.getDeclType(); }
    @Override public String getPackage() { return unboundClass.getPackage(); }
    @Override public EnumBitSet<AccessFlag> getAccessFlags() { return unboundClass.getAccessFlags(); }
    @Override public Optional<ClassType> getEnclosingClass() { return outer != null ? Optional.of(outer) : unboundClass.getEnclosingClass(); }
    @Override public Optional<Method> getEnclosingMethod() { throw new UnsupportedOperationException(); }
    @Override public Type getDescriptor() { throw new UnsupportedOperationException(); }
    @Override public String getName() { return unboundClass.getName(); }
    @Override public ClassType asRaw() { return unboundClass.asRaw(); }
    @Override public ClassType getDeclaration() { return unboundClass; }
    @Nullable public ParameterizedClass getOuter() { return outer; }
    //@formatter:on

    @Override
    public List<TypeParameter> getTypeParameters() {
        if (!arguments.isEmpty()) throw new UnsupportedOperationException();

        return unboundClass.getTypeParameters();
    }

    public boolean isFullyParameterized() {
        return getTypeArguments().size() == unboundClass.getTypeParameters().size();
    }

    @Override
    public boolean mentions(ReferenceType type) {
        return super.mentions(type) || ColUtils.anyMatch(getTypeArguments(), e -> e.mentions(type));
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == this) return true;
        if (!(obj instanceof ParameterizedClass)) return false;
        ParameterizedClass other = (ParameterizedClass) obj;

        if (getDeclaration() != other.getDeclaration()) return false;
        if (getOuter() != null && !getOuter().equals(requireNonNull(other.getOuter()))) return false;

        List<ReferenceType> args1 = getTypeArguments();
        List<ReferenceType> args2 = other.getTypeArguments();
        for (int i = 0; i < args1.size(); i++) {
            if (!args1.get(i).equals(args2.get(i))) return false;
        }

        return true;
    }

    @Override
    public int hashCode() {
        int result = outer != null ? outer.hashCode() : 0;
        result = 31 * result + unboundClass.hashCode();
        result = 31 * result + arguments.hashCode();
        return result;
    }

    @Override
    public String getFullName() {
        String args = "<" + FastStream.of(arguments).map(AType::getFullName).join(", ") + ">";
        return unboundClass.getFullName() + args;
    }
}
