package net.covers1624.coffeegrinder.bytecode.matching;

import net.covers1624.coffeegrinder.bytecode.InsnOpcode;
import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import org.jetbrains.annotations.Nullable;

import static net.covers1624.coffeegrinder.bytecode.matching.LdcMatching.matchLdcInt;
import static net.covers1624.coffeegrinder.bytecode.matching.LdcMatching.matchLdcNumber;

/**
 * Created by covers1624 on 22/11/21.
 */
public class AssignmentMatching {

    /**
     * Matches the given Instruction to a {@link CompoundAssignment} instruction.
     *
     * @param insn The Instruction to match.
     * @return The {@link CompoundAssignment} or <code>null</code>.
     */
    @Nullable
    public static CompoundAssignment matchCompoundAssignment(@Nullable Instruction insn) {
        if (insn == null || insn.opcode != InsnOpcode.COMPOUND_ASSIGNMENT) return null;

        return (CompoundAssignment) insn;
    }

    /**
     * Match the given Instruction to a {@link CompoundAssignment} of the given operation.
     *
     * @param insn The Instruction to match.
     * @param op   The Kind of operation.
     * @return The {@link CompoundAssignment} or <code>null</code>.
     */
    @Nullable
    public static CompoundAssignment matchCompoundAssignment(@Nullable Instruction insn, BinaryOp op) {
        CompoundAssignment assignment = matchCompoundAssignment(insn);
        if (assignment == null || assignment.getOp() != op) return null;

        return assignment;
    }

    /**
     * Match the given Instruction to a {@link CompoundAssignment} of the given operation
     * with the given {@link LdcNumber} value.
     *
     * @param insn The Instruction to match.
     * @param op   The Kind of operation.
     * @param ldc  The {@link LdcNumber} value.
     * @return The {@link CompoundAssignment} or <code>null</code>.
     */
    @Nullable
    public static CompoundAssignment matchCompoundAssignment(@Nullable Instruction insn, BinaryOp op, int ldc) {
        CompoundAssignment assignment = matchCompoundAssignment(insn, op);
        if (assignment == null) return null;

        if (matchLdcInt(assignment.getValue(), ldc) == null) return null;

        return assignment;
    }

    /**
     * Match the given Instruction to a {@link CompoundAssignment} type with:
     * <pre>
     *  - Either ADD or SUB operation
     *  - Int value of 1
     * </pre>
     *
     * @param insn The Instruction to match.
     * @return The {@link CompoundAssignment} or <code>null</code>.
     */
    @Nullable
    public static CompoundAssignment matchPreInc(@Nullable Instruction insn) {
        CompoundAssignment assignment = matchCompoundAssignment(insn);
        if (assignment == null) return null;
        if (assignment.getOp() != BinaryOp.ADD && assignment.getOp() != BinaryOp.SUB) return null;

        LdcNumber ldc = matchLdcNumber(assignment.getValue());
        if (ldc == null || ldc.getValue().doubleValue() != 1) return null;

        return assignment;
    }

    @Nullable
    public static Binary matchStoreArgBinaryWithPossibleCast(Store store) {
        Instruction value = store.getValue();

        if (value.opcode == InsnOpcode.CHECK_CAST) {
            Cast c = (Cast) value;
            if (c.getType() == store.getReference().getType()) {
                value = c.getArgument();
            }
        }

        if (value.opcode != InsnOpcode.BINARY) return null;
        return (Binary) value;
    }
}
