package net.covers1624.coffeegrinder.bytecode.transform.transformers.generics;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import net.covers1624.coffeegrinder.bytecode.InsnOpcode;
import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.insns.tags.InsnTag;
import net.covers1624.coffeegrinder.bytecode.transform.ClassTransformContext;
import net.covers1624.coffeegrinder.type.*;
import net.covers1624.coffeegrinder.type.TypeSubstitutions.TypeSubstApplier;
import net.covers1624.quack.collection.ColUtils;
import net.covers1624.quack.collection.FastStream;
import net.covers1624.quack.util.SneakyUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.BiConsumer;
import java.util.function.Supplier;

import static net.covers1624.quack.collection.FastStream.of;

/**
 * Created by covers1624 on 27/4/22.
 */
public class GenericTransformInference {

    public static final InsnTag REQUIRED_CAST_TAG = () -> "InferenceRequired";

    public static void infer(GenericTransform.ReturnTypeInfo ret, AbstractInvoke invoke, GenericTransform t, ClassTransformContext ctx) {
        ctx.pushStep("Infer type args for " + describe(invoke));

        // first resolve, use erased lambda return types, no nested inference
        InferenceSolution res = resolve(invoke, ret, false);

        // update lambda parameter types, and visit their return expressions. Accurate parameter types may be required for accurate return types via expression tree
        if (applyPartialSolutionToLambdas(t, res)) {
            res = resolve(invoke, ret, true);
        }

        applyResolution(invoke, res, ctx);

        applyPartialSolutionToLambdas(t, res);
        ctx.popStep();
    }

    private static boolean applyPartialSolutionToLambdas(GenericTransform t, InferenceSolution res) {
        boolean any = false;
        for (Pair<MethodDecl, ReferenceType> e : res.lambdaTypes) {
            MethodDecl lambda = e.getLeft();
            ReferenceType fType = e.getRight();
            if (!TypeSystem.areErasuresEqual(fType, lambda.getResultType())) continue;

            t.visitLambda(lambda, fType);
            any = true;
        }

        for (InferenceSolution nested : res.nested.values()) {
            any |= applyPartialSolutionToLambdas(t, nested);
        }

        return any;
    }

    private static void applyResolution(AbstractInvoke insn, InferenceSolution resolution, ClassTransformContext ctx) {
        assert !resolution.hasFailed() : resolution.failureReason;

        Method method = insn.getMethod();
        if (insn.opcode == InsnOpcode.INVOKE) {
            Invoke invoke = (Invoke) insn;

            invoke.setMethod(TypeSubstitutions.parameterize(method.getDeclaringClass(), method, resolution));
            invoke.setResultType(resolution.retType);
            invoke.explicitTypeArgs = resolution.explicit;
        } else if (insn.opcode == InsnOpcode.NEW) {
            New newInsn = (New) insn;

            ClassType type = (ClassType) resolution.retType;
            newInsn.setMethod(TypeSubstitutions.parameterize(type, method, resolution));
            newInsn.explicitClassTypeArgs = resolution.explicit;
            newInsn.explicitTypeArgs = resolution.explicit && method.hasTypeParameters();
        }

        if (resolution.polyFailed != null) {
            ctx.pushStep("Tag cast on return value as required");
            if (insn.getParent().opcode != InsnOpcode.CHECK_CAST) {
                ReferenceType castType = resolution.polyFailed;
                if (!TypeSystem.isCastableTo((ReferenceType) insn.getResultType(), castType, false)) {
                    castType = TypeSystem.erase(castType);
                }

                insn.replaceWith(new Cast(insn, castType));
            }
            insn.getParent().setTag(REQUIRED_CAST_TAG);
            ctx.popStep();
        }

        for (Map.Entry<AbstractInvoke, InferenceSolution> entry : resolution.nested.entrySet()) {
            ctx.pushStep("Infer nested " + describe(entry.getKey()));
            applyResolution(entry.getKey(), entry.getValue(), ctx);
            ctx.popStep();
        }
    }

    private static String describe(AbstractInvoke invoke) {
        if (invoke.opcode == InsnOpcode.NEW) {
            return "new " + invoke.getMethod().getDeclaringClass().getName();
        }
        return invoke.getMethod().getName();
    }

    private static InferenceSolution resolve(AbstractInvoke invoke, GenericTransform.ReturnTypeInfo ret, boolean lambdaInference) {
        GenericTransformBoundSet monoBounds = monomorphicBounds(invoke, lambdaInference);

        // without return type inference
        InferenceSolution sln = monoBounds.copy().getSolution();
        if (!BoundSet.isPoly(invoke.getMethod()) || ret.expectedType() == null) {
            return sln;
        }

        if (ret.type != null && !sln.hasRawArgs) {
            InferenceSolution polySln = addPolyRetTypeAndResolve(monoBounds.copy(), (ReferenceType) ret.type);
            if (!polySln.hasFailed()) {
                sln = polySln;
            } else {
                sln = sln.polyFailed((ReferenceType) ret.type);
            }
        }

        if (sln.hasRawArgs && monoBounds.hasParameterizedRetType()) {
            return sln; // nothing can be gained by trying to push more info from the return type into the inference
        }

        ReferenceType hintType = ret.explicitTypeHint;
        if (hintType == null) {
            // if poly inference failed and there is no explicit hint type, the result cannot be assignable
            // however we can still get a success by adding some explicit args which make the result castable
            hintType = (ReferenceType) ret.type;
        }

        AType currentRetType = sln.retType;
        if (!sln.hasFailed() && TypeSystem.isAssignableTo(currentRetType, hintType)) {
            // Explicit type hint can't possibly make things better (but could make them worse) if the result is already assignable to the hint
            return sln;
        }

        InferenceSolution explicitSln = addExplicitHintAndResolve(monoBounds, invoke.getMethod(), hintType);
        if (explicitSln.hasFailed() || explicitSln.equals(sln)) {
            return sln;
        }

        return explicitSln.makeExplicit();
    }

    private static boolean containsRaw(AType type) {
        if (type instanceof RawClass) return true;
        if (type instanceof ArrayType) return containsRaw(((ArrayType) type).getElementType());
        if (type instanceof ParameterizedClass) return of(((ParameterizedClass) type).getTypeArguments()).anyMatch(GenericTransformInference::containsRaw);
        return false;
    }

    private static boolean isRaw(AType type) {
        if (type instanceof RawClass) return true;
        if (type instanceof ArrayType) return isRaw(((ArrayType) type).getElementType());
        return false;
    }

    private static InferenceSolution addPolyRetTypeAndResolve(GenericTransformBoundSet b, ReferenceType targetType) {
        b.constrainReturnAssignable(targetType);
        return b.getSolution();
    }

    private static InferenceSolution addExplicitHintAndResolve(GenericTransformBoundSet b, Method method, ReferenceType hintType) {
        // If the ret type is parameterized, then normal poly inference should have done the best job possible
        // _unless_ the result is not assignable to the ret type and requires a cast. In this case, we can look for a common castable target type
        // and try and convert matching args on the target type, into explicit type args to make the cast better (unchecked instead of raw)
        if (b.hasParameterizedRetType()) {
            ReferenceType erasedType = (method.isConstructor() ? method.getDeclaringClass() : (ClassType) method.getReturnType()).asRaw();
            hintType = BoundSet.getHierarchyCompatibleType(erasedType, hintType);
        }
        b.explicitHint = true;
        b.constrainReturnAssignable(hintType);

        return b.getSolution();
    }

    private static GenericTransformBoundSet monomorphicBounds(AbstractInvoke invoke, boolean lambdaInference) {
        return BoundSet.monomorphicBounds(invoke, (params, retType) -> new GenericTransformBoundSet(params, retType, lambdaInference));
    }

    private static class GenericTransformBoundSet extends BoundSet {

        private final List<Pair<MethodDecl, ReferenceType>> lambdas = new LinkedList<>();
        private final boolean lambdaInference;

        public boolean explicitHint;
        @Nullable
        private OptionalBoundSource currentOption;
        @Nullable
        private Set<OptionalBoundSource> incorporating;

        public GenericTransformBoundSet(Iterable<TypeParameter> typeParameters, AType retType, boolean lambdaInference) {
            super(typeParameters, retType);
            this.lambdaInference = lambdaInference;
        }

        private GenericTransformBoundSet(GenericTransformBoundSet other) {
            super(other);
            lambdas.addAll(other.lambdas);
            lambdaInference = other.lambdaInference;
            assert other.incorporating == null;
            assert other.currentOption == null;
        }

        public boolean hasParameterizedRetType() {
            return infVarRetType instanceof ParameterizedClass;
        }

        @Override
        protected void lambdaParamsCanReceiveFunctionalInterfaceMethodType(List<AType> params, List<Parameter> fParams) {
            if (explicitHint) {
                super.lambdaParamsCanReceiveFunctionalInterfaceMethodType(params, fParams);
            }
        }

        @Override
        protected void retTypeAssignable(MethodDecl lambda, ReferenceType retType) {
            if (!lambdaInference) {
                // just do the raw assignability for now
                optional(new OptionalBoundSource(), () -> assignable((ReferenceType) lambda.getReturnType(), retType));
                return;
            }

            super.retTypeAssignable(lambda, retType);
        }

        @Override
        protected void assignable(MethodDecl lambda, ReferenceType t) {
            lambdas.add(Pair.of(lambda, t));
            super.assignable(lambda, t);
        }

        @Override
        protected void assignable(Instruction expr, ReferenceType t) {
            if (expr.opcode == InsnOpcode.CHECK_CAST) {
                assignable(((Cast) expr).getArgument(), t);
                return;
            }

            super.assignable(expr, t);
        }

        @Override
        protected void assignable(Instruction expr, ReferenceType resultType, ReferenceType t, @Nullable BoundSet polyBounds) {
            if (expr.getParent().opcode == InsnOpcode.CHECK_CAST) {
                Cast cast = (Cast) expr.getParent();
                // first thing to do is to get the args from t, into the cast
                //ReferenceType castType = inferArgsForCast((ReferenceType) cast.getType(), t);
                if (!couldBeSubtypeOf(resultType, (ReferenceType) cast.getType())) {
                    // the cast isn't going away, so lets figure out what it will be

                    if (resultType instanceof InferenceVar) {
                        assert polyBounds != null;
                        InferenceSolution r = ((GenericTransformBoundSet)polyBounds).getSolution(); // should probably somehow combine this with a polyFailed result
                        resultType = r.mapParam(((InferenceVar) resultType).param);
                    }

                    resultType = getHierarchyCompatibleType((ReferenceType) cast.getType(), resultType);
                }
            }

            if (resultType instanceof RawClass) {
                hasRawArgs = true;
            }

            if (resultType instanceof InferenceVar) {
                // todo, poly is an option
                assignable(resultType, t);
                return;
            }

            if (resultType instanceof TypeVariable) {
                // todo, optional to replace tvar with its erasure (or any supertype)
                //  unsure if this needs to be represented as an option at this stage
                assignable(resultType, t);
                return;
            }

            if (isParameterized(t)) {
                subtype(TypeSystem.erase(resultType), t);
                argsAssignableOptional(resultType, t);
                // todo, disabling those optional bounds may make the arg raw
                //   figure out when that happens, and how to propagate that info through (from nested to outer, and outer to inference)
                return;
            }

            if (t instanceof InferenceVar) {
                ReferenceType finalResultType = resultType;
                optional(new AssignmentBoundSource(resultType, t), () -> assignable(finalResultType, t));
                // don't even bother with the constraint, it needs an unchecked cast to t.param
                // TODO: this doesn't generalise very well to other kinds of generic conflicts which could be resolved with a cast to a type param
                //   but maybe, we could make a special type of optional bound, where it can be replaced with a type-param subtype, and the param is detected on conflict
                return;
            }

            assignable(resultType, t);
        }

        private boolean extendsTypeParamInScope(TypeParameter param, Instruction scope) {
            while (param.getUpperBound() instanceof TypeParameter) {
                param = (TypeParameter) param.getUpperBound();
                if (GenericTransform.typeParameterInScope(scope, param)) {
                    return true;
                }
            }

            return false;
        }

        private static boolean isParameterized(AType t) {
            return t instanceof ParameterizedClass ||
                    t instanceof ArrayType && isParameterized(((ArrayType) t).getElementType());
        }

        private void argsAssignableOptional(ReferenceType s, ReferenceType t) {
            if (t instanceof ArrayType) {
                argsAssignableOptional(
                        (ReferenceType) ((ArrayType) s).getElementType(),
                        (ReferenceType) ((ArrayType) t).getElementType());
                return;
            }

            argsAssignableOptional(s, (ParameterizedClass) t);
        }

        private void argsAssignableOptional(ReferenceType s, ParameterizedClass t) {
            ClassType tOnS = TypeSystem.findParameterizationOrNull(t.getDeclaration(), s);
            if (tOnS == null) {
                fail(s + " not a subtype of " + t);
                return;
            }
            if (tOnS instanceof RawClass) {
                return;
            }

            List<ReferenceType> tArgs = t.getTypeArguments();
            List<ReferenceType> sArgs = ((ParameterizedClass) tOnS).getTypeArguments();
            for (int i = 0; i < tArgs.size(); i++) {
                ReferenceType tArg = tArgs.get(i);
                ReferenceType sArg = sArgs.get(i);
                optional(new OptionalBoundSource(), () -> containedBy(sArg, tArg));
            }
        }

        private static class Result {

            private final Map<InferenceVar, ReferenceType> sln;
            private final double cost;

            public Result(Map<InferenceVar, ReferenceType> sln, double cost) {
                this.sln = sln;
                this.cost = cost;
            }
        }

        public InferenceSolution getSolution() {
            if (failure != null) return InferenceSolution.failure(failure);

            try {
                return getSolution(solve());
            } catch (ResolveFailedException ex) {
                return InferenceSolution.failure(ex.getMessage());
            }
        }

        private InferenceSolution getSolution(InferenceVarMapper mapper) {
            AType retType = TypeSubstitutions.subst(infVarRetType, mapper);
            if (retType instanceof ReferenceType && hasRawArgs)
                retType = TypeSystem.erase((ReferenceType) retType);

            InferenceSolution sln = InferenceSolution.success(
                    of(vars.entrySet()).toImmutableMap(Map.Entry::getKey, e -> mapper.mapParam(e.getValue())),
                    retType,
                    of(nestedVars.entrySet()).toImmutableMap(Map.Entry::getKey, e -> ((GenericTransformBoundSet) e.getValue()).getSolution(mapper)),
                    of(lambdas).map(e -> Pair.of(e.getLeft(), TypeSubstitutions.subst(e.getRight(), mapper))).toImmutableList(),
                    hasRawArgs
            );

            return sln;
        }

        @Override
        protected Map<BoundSet.InferenceVar, ReferenceType> solveVars(List<BoundSet.InferenceVar> vars, BoundSet.InferenceVarMapper solved) {
            List<OptionalBoundSource> opts = of(vars)
                    .flatMap(var -> boundsFor(var).allOpts())
                    .distinct()
                    .filter(OptionalBoundSource::canDisable)
                    .toImmutableList();
            return tryDisablingOptions(ImmutableList.of(), opts, () -> super.solveVars(vars, solved)).sln;
        }

        // [], [A, B, C, D]
        //   [D], []
        //   [C], [D]...
        //     [C, D], []
        //   [B], [C, D]
        //     [B, D], []
        //     [B, C], [D]
        //        [B, C, D], []
        //   [A], [B, C, D]
        //     [A, D], []
        //     [A, C], [D]
        //       [A, C, D], []
        //     [A, B], [C, D]
        //       [A, B, D], []
        //       [A, B, C], [D]
        //         [A, B, C, D], []
        private Result tryDisablingOptions(List<OptionalBoundSource> disabled, List<OptionalBoundSource> remaining, Supplier<Map<InferenceVar, ReferenceType>> resultImpl) {
            try {
                setDisabled(disabled, true);
                try {
                    return new Result(resultImpl.get(), FastStream.of(disabled).doubleSum(OptionalBoundSource::cost));
                } catch (ResolveFailedException ex) {
                    // Throw if we don't have any more options to disable.
                    if (remaining.isEmpty()) {
                        throw ex;
                    }
                }
            } finally {
                setDisabled(disabled, false);
            }

            Result best = null;
            for (int i = remaining.size() - 1; i >= 0; i--) {
                OptionalBoundSource next = remaining.get(i);
                try {
                    Result r = tryDisablingOptions(concat(disabled, next), remaining.subList(i + 1, remaining.size()), resultImpl);
                    if (best == null || r.cost <= best.cost) {
                        best = r;
                    }
                } catch (ResolveFailedException ex) {
                    if (i == 0 && best == null) {
                        throw ex;
                    }
                }
            }

            assert best != null;
            return best;
        }

        private static <T> List<T> concat(List<T> list, T other) {
            ImmutableList.Builder<T> builder = ImmutableList.builder();
            builder.addAll(list);
            builder.add(other);
            return builder.build();
        }

        private void setDisabled(Iterable<OptionalBoundSource> opts, boolean disabled) {
            for (OptionalBoundSource opt : opts) {
                for (Bound bound : opt.bounds) {
                    bound.disabled = disabled;
                }
            }
        }

        private void optional(OptionalBoundSource opt, Runnable action) {
            OptionalBoundSource prev = currentOption;
            currentOption = opt;
            try {
                action.run();
            } finally {
                currentOption = prev;
            }
        }

        private boolean couldBeSubtypeOf(ReferenceType s, ReferenceType t) {
            if (s instanceof InferenceVar) {
                DetailedVarBounds b = boundsFor((InferenceVar) s);
                return b.equalTypes(TypeSubstApplier.NONE).allMatch(e -> e instanceof InferenceVar || couldBeSubtypeOf(e, t))
                        && b.lowerTypes(TypeSubstApplier.NONE).allMatch(l -> l instanceof InferenceVar || couldBeSubtypeOf(l, t));
            }

            if (t instanceof ParameterizedClass) {
                ParameterizedClass tParam = (ParameterizedClass) t;
                ClassType tOnS = TypeSystem.findParameterizationOrNull(tParam.getDeclaration(), s);
                if (tOnS == null) {
                    return false;
                }
                if (tOnS instanceof RawClass) {
                    return true;
                }

                List<ReferenceType> tArgs = tParam.getTypeArguments();
                List<ReferenceType> sArgs = ((ParameterizedClass) tOnS).getTypeArguments();
                for (int i = 0; i < tArgs.size(); i++) {
                    if (!couldBeContainedBy(sArgs.get(i), tArgs.get(i))) { return false; }
                }

                return true;
            }

            return TypeSystem.isAssignableTo(s, t);
        }

        private boolean couldBeContainedBy(ReferenceType s, ReferenceType t) {
            if (t instanceof InferenceVar) {
                DetailedVarBounds b = boundsFor((InferenceVar) t);
                return b.equalTypes(TypeSubstApplier.NONE).allMatch(e -> e instanceof InferenceVar || couldBeEqual(e, s))
                        && b.upperTypes(TypeSubstApplier.NONE).allMatch(u -> u instanceof InferenceVar || couldBeSubtypeOf(s, u))
                        && b.lowerTypes(TypeSubstApplier.NONE).allMatch(l -> l instanceof InferenceVar || couldBeSubtypeOf(l, s));
            }
            if (s instanceof InferenceVar) {
                DetailedVarBounds b = boundsFor((InferenceVar) s);
                return b.equalTypes(TypeSubstApplier.NONE).allMatch(e -> e instanceof InferenceVar || couldBeContainedBy(e, t))
                        && b.upperTypes(TypeSubstApplier.NONE).allMatch(u -> u instanceof InferenceVar || couldBeContainedBy(u, t))
                        && b.lowerTypes(TypeSubstApplier.NONE).allMatch(l -> l instanceof InferenceVar || couldBeContainedBy(l, t));
            }

            if (!(t instanceof WildcardType)) {
                return couldBeEqual(s, t);
            }

            if (((WildcardType) t).isSuper()) {
                return couldBeSubtypeOf(t.getLowerBound(), s.getLowerBound());
            }

            if (!(s instanceof WildcardType)) {
                return couldBeSubtypeOf(s, t.getUpperBound());
            }

            if (((WildcardType) s).isSuper()) {
                return couldBeEqual(TypeSystem.objectType(s), t.getUpperBound());
            }
            return couldBeSubtypeOf(s.getUpperBound(), t.getUpperBound());
        }

        private boolean couldBeEqual(ReferenceType t1, ReferenceType t2) {
            if (t1 instanceof InferenceVar) {
                return couldBeContainedBy(t2, t1);
            }
            if (t2 instanceof InferenceVar) {
                return couldBeEqual(t2, t1);
            }

            // need to recurse on type args :|
            return t1.equals(t2);
        }

        @Override
        protected void failProperSubtype(ReferenceType s, ReferenceType t) {
            if (incorporating != null && ColUtils.anyMatch(incorporating, opt -> handleSubtypeIncorporation(opt, s, t))) {
                return;
            }

            super.failProperSubtype(s, t);
        }

        private boolean handleSubtypeIncorporation(OptionalBoundSource opt, ReferenceType s, ReferenceType t) {
            if (opt instanceof AssignmentBoundSource) {
                AssignmentBoundSource cOpt = (AssignmentBoundSource) opt;
                if (s != cOpt.s) return false;

                // clearly, s has a conflict with something, maybe by looking at that conflict, we could find something else to cast s to
                if (t instanceof TypeParameter && TypeSystem.areErasuresEqual(s, t)) {
                    // looks like s could have been the erasure of a cast to t (type parameter), lets try re-running the assignability check with T instead
                    // we should probably  be also checking if T is in scope

                    if (!cOpt.alternatives.contains(t)) {
                        assert cOpt.alternatives.isEmpty();
                        cOpt.alternatives.add(t);
                        // We could add other optional bounds which represent the 'alternatives'
                        // But until we find a motivating test case, just disabling this bound completely is fine.
                        // We know the engine can find at least some of the information we were trying to provide with this bound elsewhere, because that's the source of the conflict

                        //OptionalBoundSource opt2 = new OptionalBoundSource();
                        //optional(opt2, () -> assignable(t, cOpt.t));
                    }
                    return true;
                }
            }

            return false;
        }

        @Override
        protected void fail(String reason) {
            if (incorporating != null) {
                if (!incorporating.isEmpty()) {
                    return; // todo, report incompatibility, make searching option space faster?
                }
            } else if (currentOption != null) {
                // todo, report failure against option, force disabled
                return;
            }

            super.fail(reason);
        }

        private Set<OptionalBoundSource> getBoundSources() {
            if (incorporating != null) return incorporating;
            if (currentOption != null) return ImmutableSet.of(currentOption);
            return ImmutableSet.of();
        }

        // @formatter:off
        @Override protected DetailedVarBounds newVarBounds() { return new DetailedVarBounds(this); }
        @Override protected DetailedVarBounds copyVarBounds(BoundSet.VarBounds<?> other) { return new DetailedVarBounds(this, (DetailedVarBounds) other); }
        @Override protected DetailedVarBounds boundsFor(InferenceVar var) { return (DetailedVarBounds) super.boundsFor(var); }
        @Override protected DetailedVarBounds boundsFor(TypeParameter param) { return (DetailedVarBounds) super.boundsFor(param); }
        @Override protected GenericTransformBoundSet makeNestedBoundSet(Iterable<TypeParameter> vars, AType retType) { return new GenericTransformBoundSet(vars, retType, false); }
        public GenericTransformBoundSet copy() { return new GenericTransformBoundSet(this); }
        // @formatter:on

        private static class OptionalBoundSource {

            private static final double CAPTURE_COST = 0.01;
            private static final double DEFAULT_COST = 1;

            List<Bound> bounds = new ArrayList<>(4);

            public double cost() {
                if (ColUtils.allMatch(bounds, b -> b.type instanceof CapturedTypeVar)) { return CAPTURE_COST; }

                return DEFAULT_COST;
            }

            @Override
            public String toString() {
                return String.format("opt#%1$04X", hashCode() & 0xFFFF);
            }

            public boolean canDisable() {
                return true;
            }
        }

        private static class AssignmentBoundSource extends OptionalBoundSource {

            public final ReferenceType s;
            public final ReferenceType t;
            public LinkedList<ReferenceType> alternatives = new LinkedList<>();

            public AssignmentBoundSource(ReferenceType s, ReferenceType t) {
                this.s = s;
                this.t = t;
            }

            @Override
            public boolean canDisable() {
                return !alternatives.isEmpty();
            }
        }

        private static class Bound {

            public final ReferenceType type;
            public final Set<OptionalBoundSource> opts;
            public boolean disabled;

            public Bound(ReferenceType type, Set<OptionalBoundSource> opts) {
                this.type = type;
                this.opts = opts;
            }

            @Override
            public boolean equals(Object o) {
                if (this == o) return true;
                if (!(o instanceof Bound)) return false;
                Bound other = (Bound) o;

                return type.equals(other.type) && opts.equals(other.opts);
            }

            @Override
            public String toString() {
                if (opts.isEmpty()) return type.toString();

                return "{" + of(opts).join(", ") + "} " + type;
            }
        }

        private static class DetailedVarBounds extends BoundSet.VarBounds<Bound> {

            public DetailedVarBounds(BoundSet root) {
                super(root);
            }

            public DetailedVarBounds(BoundSet root, DetailedVarBounds other) {
                super(root, other);
            }

            @Override
            public GenericTransformBoundSet getRoot() {
                return (GenericTransformBoundSet) super.getRoot();
            }

            @Override
            protected Bound makeBound(ReferenceType t) {
                return new Bound(t, getRoot().getBoundSources());
            }

            @Override
            protected ReferenceType getType(Bound bound) {
                return bound.type;
            }

            @Override
            protected void boundAdded(Bound b) {
                for (OptionalBoundSource opt : b.opts) {
                    opt.bounds.add(b);
                }
            }

            @Override
            protected boolean isEncompassedBy(Bound b1, Bound b2) {
                return b1.type.equals(b2.type) && b1.opts.containsAll(b2.opts);
            }

            @Override
            protected <T1, T2> void incorporate(T1 b1, T2 b2, BiConsumer<T1, T2> action) {
                Set<OptionalBoundSource> prev = getRoot().incorporating;

                Set<OptionalBoundSource> combined = ImmutableSet.of();
                if (b1 instanceof Bound) combined = Sets.union(combined, ((Bound) b1).opts);
                if (b2 instanceof Bound) combined = Sets.union(combined, ((Bound) b2).opts);
                getRoot().incorporating = combined;

                try {
                    action.accept(b1, b2);
                } finally {
                    getRoot().incorporating = prev;
                }
            }

            @Override
            protected FastStream<Bound> activeBounds(List<Bound> list) {
                return SneakyUtils.<FastStream<Bound>>unsafeCast(super.activeBounds(list))
                        .filter(e -> !e.disabled);
            }

            public FastStream<OptionalBoundSource> allOpts() {
                return FastStream.of(equalBounds, lowerBounds, upperBounds)
                        .flatMap(this::activeBounds)
                        .flatMap(e -> e.opts);
            }
        }
    }
}
