package net.covers1624.coffeegrinder.util;

import net.covers1624.quack.util.Copyable;
import net.covers1624.quack.util.SneakyUtils;

import java.util.BitSet;
import java.util.EnumSet;

/**
 * A {@link BitSet} specialized for Enum values effectively creating
 * a Enum Bitfield.
 * <p>
 * Created by covers1624 on 20/4/21.
 */
public class EnumBitSet<E extends Enum<E>> extends BitSet implements Copyable<EnumBitSet<E>> {

    private final Class<E> elementType;

    private EnumBitSet(Class<E> clazz) {
        super(clazz.getEnumConstants().length);
        this.elementType = clazz;
    }

    public static <T extends Enum<T>> EnumBitSet<T> noneOf(Class<T> clazz) {
        return new EnumBitSet<>(clazz);
    }

    @SafeVarargs
    public static <T extends Enum<T>> EnumBitSet<T> of(T... elements) {
        if (elements.length == 0) throw new IllegalArgumentException("Elements should not be empty");
        T first = elements[0];
        EnumBitSet<T> set = noneOf(first.getDeclaringClass());
        for (T element : elements) {
            set.set(element);
        }
        return set;
    }

    public static <T extends Enum<T>> EnumBitSet<T> copyOf(EnumBitSet<T> other) {
        EnumBitSet<T> copy = noneOf(other.elementType);
        for (T e : other.elementType.getEnumConstants()) {
            if (other.get(e)) {
                copy.set(e);
            }
        }
        return copy;
    }

    public void flip(E bit) {
        super.flip(bit.ordinal());
    }

    public boolean get(E bit) {
        return super.get(bit.ordinal());
    }

    public void set(E bit) {
        super.set(bit.ordinal());
    }

    public void clear(E bit) {
        super.clear(bit.ordinal());
    }

    public EnumSet<E> toSet() {
        EnumSet<E> set = EnumSet.noneOf(elementType);
        for (E e : elementType.getEnumConstants()) {
            if (get(e)) {
                set.add(e);
            }
        }
        return set;
    }

    @Override
    public EnumBitSet<E> clone() {
        return SneakyUtils.unsafeCast(super.clone());
    }

    @Override
    public EnumBitSet<E> copy() {
        return copyOf(this);
    }

    @Override
    public String toString() {
        return toSet().toString();
    }
}
