package net.covers1624.coffeegrinder.bytecode.flow;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;

/**
 * Created by covers1624 on 20/4/21.
 */
public class Dominance {

    /**
     * Computes the dominator tree of a ControlFlow graph.
     * <p>
     * Precondition: the dominance tree is not already computed for some nodes reachable from entryPoint
     * (i.e. ImmediateDominator and DominatorTreeChildren are both null),
     * and the visited flag is false for any nodes reachable from entryPoint.
     * <p>
     * Postcondition: a dominator tree is constructed for all nodes reachable from entryPoint,
     * and the visited flag remains false.
     *
     * @param entryPoint The entry point.
     */
    public static void computeDominance(ControlFlowNode entryPoint) {
        // A Simple, Fast Dominance Algorithm
        // Keith D. Cooper, Timothy J. Harvey and Ken Kennedy

        List<ControlFlowNode> nodes = new ArrayList<>();
        entryPoint.traversePostOrder(ControlFlowNode::getSuccessors, nodes::add);
        assert nodes.get(nodes.size() - 1) == entryPoint;
        for (int i = 0; i < nodes.size(); i++) {
            nodes.get(i).postOrderNumber = i;
        }

        // For the purpose of this algorithm, make the entry point its own dominator.
        // We'll reset it back to null at the end of this function.
        entryPoint.immediateDominator = entryPoint;

        boolean changed;
        do {
            changed = false;

            // For all nodes b except the entry point (in reverse post-order)
            for (int i = nodes.size() - 2; i >= 0; i--) {
                ControlFlowNode b = nodes.get(i);
                // Compute new immediate dominator.
                ControlFlowNode newIdom = null;
                for (ControlFlowNode p : b.getPredecessors()) {
                    // Ignore predecessors that were not processed yet
                    if (p.immediateDominator != null) {
                        if (newIdom == null) {
                            newIdom = p;
                        } else {
                            newIdom = findCommonDominator(p, newIdom);
                        }
                    }
                }
                // The reverse post-order ensures at least one of our predecessors was processed.
                assert newIdom != null;
                if (newIdom != b.immediateDominator) {
                    b.immediateDominator = newIdom;
                    changed = true;
                }
            }

        }
        while (changed);
        // Create dominator tree for all reachable nodes:
        for (ControlFlowNode node : nodes) {
            if (node.immediateDominator != null) {
                node.dominatorTreeChildren = new ArrayList<>();
            }
        }
        entryPoint.immediateDominator = null;
        for (ControlFlowNode node : nodes) {
            // Create a list of children in the dominator tree
            if (node.immediateDominator != null) {
                assert node.immediateDominator.dominatorTreeChildren != null;
                node.immediateDominator.dominatorTreeChildren.add(node);
            }
            //Also reset the visited flag.
            node.visited = false;
        }
    }

    /**
     * Returns the common ancestor of a and b in the dominator tree.
     * Both a and b must be part of the same dominator tree.
     *
     * @param a The a node.
     * @param b The b node.
     * @return The common ancestor.
     */
    public static ControlFlowNode findCommonDominator(ControlFlowNode a, ControlFlowNode b) {
        while (a != b) {
            while (a.postOrderNumber < b.postOrderNumber) {
                a = a.getImmediateDominator();
            }
            while (b.postOrderNumber < a.postOrderNumber) {
                b = b.getImmediateDominator();
            }
        }
        return a;
    }

    /**
     * Computes a bitset where
     * <code>bits.get(i) == true</code> if <code>cfg[i]</code> is reachable and there is some node
     * that is reachable from <code>cfg[i]</code> but not dominated by <code>cfg[i]</code>.
     * <p>
     * This is similar to "does cfg[i] have a non-empty dominance frontier?"
     * except that it uses non-strict dominance where the definition of dominance frontiers
     * uses "strictly dominates".
     * <p>
     * Precondition:
     * - Dominance was computed for cfg and <code>cfg[i].cfgIndex == i</code> for all i.
     *
     * @param bits The bitset to store the exits in.
     * @param cfg  The CFG.
     */
    public static void computeReachableExits(BitSet bits, ControlFlowNode[] cfg) {
        assert assertNodeIndex(cfg);

        for (ControlFlowNode j : cfg) {
            // If j is a join-point (more than one incoming node):
            // `j.isReachable() && j.immediateDominator == null` is the root node, which counts as an extra incoming edge
            if (j.isReachable() && (j.getPredecessors().size() >= 2 || (j.getPredecessors().size() >= 1 && j.immediateDominator == null))) {
                // Add j to frontier of all predecessors and their dominators up to j's immediate dominator.
                for (ControlFlowNode p : j.getPredecessors()) {
                    for (ControlFlowNode runner = p; runner != j.immediateDominator && runner != j && runner != null; runner = runner.immediateDominator) {
                        bits.set(runner.cfgIndex);
                    }
                }
            }
        }
    }

    //Java assertion hax, this method is only run when assertions are enabled.
    private static boolean assertNodeIndex(ControlFlowNode[] cfg) {
        for (int i = 0; i < cfg.length; i++) {
            assert cfg[i].cfgIndex == i;
        }
        return true;
    }
}
