package net.covers1624.coffeegrinder.bytecode.insns;

import net.covers1624.coffeegrinder.bytecode.InsnVisitor;
import net.covers1624.coffeegrinder.bytecode.InstructionFlag;
import net.covers1624.coffeegrinder.type.AType;
import net.covers1624.coffeegrinder.type.PrimitiveType;
import net.covers1624.coffeegrinder.util.EnumBitSet;
import org.jetbrains.annotations.Nullable;

import java.util.Objects;

import static net.covers1624.coffeegrinder.bytecode.InstructionFlag.END_POINT_UNREACHABLE;
import static net.covers1624.coffeegrinder.bytecode.InstructionFlag.MAY_BRANCH;

/**
 * Unconditional branch.
 * <p>
 * When humping to the entrypoint of the current block container, the branch represents a <code>continue</code> statement.
 * <p>
 * Created by covers1624 on 25/2/21.
 */
public final class Branch extends SimpleInstruction {

    private static final EnumBitSet<InstructionFlag> FLAGS = EnumBitSet.of(END_POINT_UNREACHABLE, MAY_BRANCH);

    private Block targetBlock;

    public Branch(Block targetBlock) {
        this.targetBlock = targetBlock;
    }

    @Override
    protected void onConnected() {
        super.onConnected();
        targetBlock.addBranch(this);
    }

    @Override
    protected void onDisconnected() {
        super.onDisconnected();
        targetBlock.remBranch(this);
    }

    @Override
    public AType getResultType() {
        return PrimitiveType.VOID;
    }

    @Override
    public EnumBitSet<InstructionFlag> getDirectFlags() {
        return FLAGS;
    }

    @Override
    public <R, C> R accept(InsnVisitor<R, C> visitor, C ctx) {
        return visitor.visitBranch(this, ctx);
    }

    @Override
    public Branch copy() {
        return new Branch(targetBlock);
    }

    //@formatter:off
    @Nullable public BlockContainer getTargetContainerOrNull() { return (BlockContainer) targetBlock.getParentOrNull(); }
    public BlockContainer getTargetContainer() { return Objects.requireNonNull(getTargetContainerOrNull()); }
    public Block getTargetBlock() { return targetBlock; }
    //@formatter:on

    public void setTargetBlock(Block targetBlock) {
        if (isConnected()) {
            this.targetBlock.remBranch(this);
        }
        this.targetBlock = targetBlock;
        if (isConnected()) {
            this.targetBlock.addBranch(this);
        }
    }
}
