package net.covers1624.coffeegrinder.bytecode.insns;

import net.covers1624.coffeegrinder.bytecode.*;
import net.covers1624.coffeegrinder.type.*;
import net.covers1624.coffeegrinder.util.EnumBitSet;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;
import org.objectweb.asm.Type;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import static java.util.Objects.requireNonNull;
import static net.covers1624.quack.util.SneakyUtils.unsafeCast;

/**
 * Created by covers1624 on 4/5/21.
 */
public final class ClassDecl extends Instruction {

    private ClassType clazz;

    public final InstructionCollection<Instruction> members = new InstructionCollection<>(this);

    // Populated by RecordTransformer
    public final List<RecordComponentDecl> recordComponents = new ArrayList<>();
    public @Nullable MethodDecl canonicalCtor;

    @Nullable
    private Map<Method, MethodDecl> methodLookup;
    @Nullable
    private List<ClassDecl> nestedClasses;

    public ClassDecl(ClassType clazz) {
        this.clazz = clazz;
    }

    @Override
    public AType getResultType() {
        return PrimitiveType.VOID;
    }

    @Override
    public EnumBitSet<InstructionFlag> getDirectFlags() {
        return InstructionFlag.NONE;
    }

    @Override
    public <R, C> R accept(InsnVisitor<R, C> visitor, C ctx) {
        return visitor.visitClassDecl(this, ctx);
    }

    //@formatter:off
    public ClassType getClazz() { return clazz; }
    public FastStream<FieldDecl> getFieldMembers() { return unsafeCast(members.filter(e -> e instanceof FieldDecl)); }
    public FastStream<MethodDecl> getMethodMembers() { return unsafeCast(members.filter(e -> e instanceof MethodDecl)); }
    public FastStream<ClassDecl> getClassMembers() { return unsafeCast(members.filter(e -> e instanceof ClassDecl)); }
    public void setClazz(ClassType clazz) { this.clazz = clazz; }
    //@formatter:on

    @Nullable
    public FieldDecl findField(String name, Type desc) {
        return getFieldMembers().filter(e -> e.getField().getName().equals(name) && e.getField().getDescriptor().equals(desc)).onlyOrDefault();
    }

    @Nullable
    public FieldDecl findField(Field field) {
        return getFieldMembers().filter(e -> e.getField() == field).onlyOrDefault();
    }

    public FieldDecl getField(String name, Type desc) {
        return requireNonNull(findField(name, desc));
    }

    public void onParsed() {
        if (methodLookup != null) throw new IllegalStateException("Internal method.");
        methodLookup = getMethodMembers().toImmutableMap(MethodDecl::getMethod, Function.identity());
        nestedClasses = getClassMembers().toImmutableList();
    }

    public List<ClassDecl> getNestedClasses() {
        return requireNonNull(nestedClasses, "Only available after methods have finished being parsed.");
    }

    public Map<Method, MethodDecl> getMethodLookup() {
        return requireNonNull(methodLookup, "Only available after methods have finished being parsed.");
    }

    @Nullable
    public MethodDecl findMethod(String name, Type desc) {
        return FastStream.of(getMethodLookup().values())
                .filter(e -> e.getMethod().getName().equals(name) && e.getMethod().getDescriptor().equals(desc))
                .onlyOrDefault();
    }

    @Nullable
    public MethodDecl findMethod(Method method) {
        return getMethodLookup().get(method);
    }

    public MethodDecl getMethod(String name, Type desc) {
        return requireNonNull(findMethod(name, desc));
    }

    public MethodDecl getMethod(Method method) {
        return requireNonNull(findMethod(method));
    }

    public static class RecordComponentDecl {

        public final FieldDecl field;
        public final boolean isVarargs;

        // May be null if the accessor was stripped
        public @Nullable MethodDecl accessor;

        public final TypeAnnotationData typeAnnotations;
        public final List<AnnotationData> regularAnnotations;

        public RecordComponentDecl(FieldDecl field, boolean isVarargs, MethodDecl accessor, TypeAnnotationData typeAnnotations, List<AnnotationData> regularAnnotations) {
            this.field = field;
            this.isVarargs = isVarargs;
            this.accessor = accessor;
            this.typeAnnotations = typeAnnotations;
            this.regularAnnotations = regularAnnotations;
        }
    }
}
