package net.covers1624.coffeegrinder.bytecode.insns;

import net.covers1624.coffeegrinder.bytecode.*;
import net.covers1624.coffeegrinder.type.AType;
import net.covers1624.coffeegrinder.type.Method;
import net.covers1624.coffeegrinder.type.ReferenceType;
import net.covers1624.coffeegrinder.type.TypeSystem;
import net.covers1624.coffeegrinder.util.EnumBitSet;
import org.jetbrains.annotations.Nullable;

import java.util.*;

/**
 * Created by covers1624 on 19/4/21.
 */
public final class MethodDecl extends Instruction {

    public final IndexedInstructionCollection<ParameterVariable> parameters = new IndexedInstructionCollection<>(this);
    public final InstructionCollection<LocalVariable> variables = new InstructionCollection<>(this);

    private final Method method;

    private final InstructionSlot<Instruction> body = new InstructionSlot<>(this);
    private AType returnType;

    @Nullable
    private ReferenceType resultType;

    private final Map<Block, String> blockNames = new HashMap<>();

    private final List<Return> returns = new LinkedList<>();

    public MethodDecl(Method method, Instruction body, List<ParameterVariable> parameters) {
        this.method = method;
        this.parameters.setValues(parameters);
        this.body.set(body);
        returnType = method.getReturnType();
    }

    @Override
    public ReferenceType getResultType() {
        return Objects.requireNonNull(resultType, "Result type only available for lambdas.");
    }

    @Override
    protected EnumBitSet<InstructionFlag> computeFlags() {
        if (getParent() instanceof ClassDecl) {
            // Methods currently don't expose any flags to the class.
            return InstructionFlag.NONE;
        }
        // Lambdas should be treated like a NewObject insn.
        return InstructionFlag.MAY_THROW.toSet();
    }

    @Override
    public EnumBitSet<InstructionFlag> getDirectFlags() {
        return InstructionFlag.NONE;
    }

    @Override
    public <R, C> R accept(InsnVisitor<R, C> visitor, C ctx) {
        return visitor.visitMethodDecl(this, ctx);
    }

    //@formatter:off
    public Method getMethod() { return method; }
    public AType getReturnType() { return returnType; }
    public void setBody(Instruction body) { this.body.set(body); }
    public void setReturnType(AType returnType) { this.returnType = returnType; }
    void addReturn(Return ret) { returns.add(ret); }
    void removeReturn(Return ret) { returns.remove(ret); }
    //@formatter:on

    public void setResultType(ReferenceType resultType) {
        assert resultType.getFunctionalInterfaceMethod() != null;
        assert this.resultType == null || TypeSystem.areErasuresEqual(resultType, this.resultType);
        this.resultType = resultType;
    }

    public List<Return> getReturns() {
        return returns;
    }

    public boolean hasBody() {
        return !(body.get() instanceof Nop);
    }

    public BlockContainer getBody() {
        if (!hasBody()) throw new UnsupportedOperationException("This function has no body.");
        return (BlockContainer) body.get();
    }

    String getSynBlockName(Block block) {
        String name = blockNames.get(block);
        if (name == null) {
            name = "SYN_L" + blockNames.size();
            blockNames.put(block, name);
        }
        return name;
    }
}
