package net.covers1624.coffeegrinder.bytecode.matching;

import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import org.jetbrains.annotations.Nullable;

import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.matchLoadLocal;

/**
 * Created by covers1624 on 19/4/21.
 */
public class BranchLeaveMatching {

    /**
     * Attempts to match a {@link Return} instruction that is exiting a function.
     *
     * @param insn The instruction.
     * @return The {@link Return} instruction, otherwise <code>null</code>.
     */
    @Nullable
    public static Return matchReturn(@Nullable Instruction insn) {
        if (!(insn instanceof Return ret)) return null;

        return ret;
    }

    @Nullable
    public static Return matchReturn(@Nullable Instruction insn, MethodDecl method) {
        if (!(insn instanceof Return ret)) return null;

        return ret.getMethod() == method ? ret : null;
    }

    /**
     * Attempts to match a leave instruction.
     *
     * @param insn The instruction.
     * @return The Leave instruction, otherwise empty.
     */
    @Nullable
    public static Leave matchLeave(@Nullable Instruction insn) {
        if (!(insn instanceof Leave leave)) return null;

        return leave;
    }

    /**
     * Attempts to match a leave instruction which leaves the specified BlockContainer.
     *
     * @param insn      The instruction.
     * @param container The BlockContainer.
     * @return The Leave instruction, otherwise empty.
     */
    @Nullable
    public static Leave matchLeave(@Nullable Instruction insn, BlockContainer container) {
        Leave leave = matchLeave(insn);
        if (leave == null) return null;

        if (leave.getTargetContainer() != container) return null;
        return leave;
    }

    /**
     * Attempts to match a branch instruction.
     *
     * @param insn The instruction.
     * @return The Branch instruction, otherwise empty.
     */
    @Nullable
    public static Branch matchBranch(@Nullable Instruction insn) {
        if (!(insn instanceof Branch branch)) return null;

        return branch;
    }

    /**
     * Attempts to match a branch instruction to a given block.
     *
     * @param insn        The instruction.
     * @param targetBlock The target block.
     * @return The Branch instruction, otherwise empty.
     */
    @Nullable
    public static Branch matchBranch(@Nullable Instruction insn, Block targetBlock) {
        Branch branch = matchBranch(insn);
        if (branch == null) return null;
        if (branch.getTargetBlock() != targetBlock) return null;
        return branch;
    }

    /**
     * Returns true if exit1 and exit2 are both exit instructions
     * (branch or leave) and both represent the same exit.
     *
     * @param exit1 The first exit point.
     * @param exit2 The second exit point.
     * @return If they represent the same exit.
     */
    public static boolean compatibleExitInstruction(@Nullable Instruction exit1, @Nullable Instruction exit2) {
        return switch (exit1) {
            case Branch br1 when exit2 instanceof Branch br2 -> br1.getTargetBlock() == br2.getTargetBlock();
            case Leave l1 when exit2 instanceof Leave l2 -> l1.getTargetContainer() == l2.getTargetContainer();
            case null, default -> false;
        };
    }

    /**
     * Attempts to match a {@link Throw} instruction.
     *
     * @param insn The Instruction.
     * @return The {@link Throw} or <code>null</code>.
     */
    @Nullable
    public static Throw matchThrow(@Nullable Instruction insn) {
        if (!(insn instanceof Throw thr)) return null;

        return thr;
    }

    /**
     * Attempts to match a {@link Throw} instruction, which loads
     * the specified variable.
     *
     * @param insn     The Instruction.
     * @param variable The variable.
     * @return The {@link Throw} or <code>null</code>.
     */
    @Nullable
    public static Throw matchThrow(@Nullable Instruction insn, LocalVariable variable) {
        Throw thr = matchThrow(insn);
        if (thr == null) return null;

        if (matchLoadLocal(thr.getArgument(), variable) == null) return null;

        return thr;
    }
}
