package net.covers1624.coffeegrinder.bytecode.transform.transformers;

import net.covers1624.coffeegrinder.bytecode.AccessFlag;
import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.insns.Invoke.InvokeKind;
import net.covers1624.coffeegrinder.bytecode.matching.BranchLeaveMatching;
import net.covers1624.coffeegrinder.bytecode.transform.ClassTransformContext;
import net.covers1624.coffeegrinder.bytecode.transform.ClassTransformer;
import net.covers1624.coffeegrinder.type.ClassType;
import net.covers1624.coffeegrinder.type.Parameter;
import net.covers1624.coffeegrinder.type.TypeSystem;
import net.covers1624.quack.collection.ColUtils;
import org.objectweb.asm.Type;

import java.util.List;

import static net.covers1624.coffeegrinder.bytecode.matching.InvokeMatching.getSuperConstructorCall;
import static net.covers1624.coffeegrinder.bytecode.matching.InvokeMatching.matchInvoke;

/**
 * We have the following cases to deal with:
 * <pre>
 * - Remove calls to super with no arguments.
 * - Nuke anon class passthrough constructors.
 * - Remove constructors which only have a single super call
 *   - If there are no other constructors and
 *   - If the method is declared as public or the class is an enum
 * - Remove empty static initializers
 * </pre>
 * Created by covers1624 on 26/11/21.
 */
public class ImplicitConstructorCleanup implements ClassTransformer {

    @SuppressWarnings ("NotNullFieldNotInitialized")
    private ClassTransformContext ctx;

    @Override
    public void transform(ClassDecl cInsn, ClassTransformContext ctx) {
        this.ctx = ctx;

        if (cInsn.getClazz().getDeclType() == ClassType.DeclType.TOP_LEVEL) {
            apply(cInsn);
        }
    }

    private void apply(ClassDecl cInsn) {
        ClassType cType = cInsn.getClazz();
        List<MethodDecl> constructors = cInsn.getMethodMembers()
                .filter(e -> e.getMethod().isConstructor())
                .toLinkedList();

        ctx.pushStep("Cleanup targeted inner constructor calls " + cType.getName());
        for (MethodDecl constructor : constructors) {
            cleanupTargetedInnerConstructorCall(constructor);
        }
        ctx.popStep();

        ctx.pushStep("Remove redundant super calls " + cType.getName());
        for (MethodDecl constructor : constructors) {
            removeRedundantSuperCall(constructor);
        }
        ctx.popStep();

        if (cInsn.getClazz().getAccessFlags().get(AccessFlag.RECORD)) {
            ctx.pushStep("Cleanup redundant canonical record constructors " + cType.getName());
            removeRedundantCanonicalRecordConstructor(cInsn);
            ctx.popStep();
        } else {
            ctx.pushStep("Remove single empty constructor " + cType.getName());
            removeRedundantConstructors(cInsn, constructors);
            ctx.popStep();
        }

        MethodDecl staticInit = cInsn.findMethod("<clinit>", Type.getMethodType("()V"));
        if (staticInit != null && isEmpty(staticInit)) {
            ctx.pushStep("Remove empty static initializer " + cType.getName());
            staticInit.remove();
            ctx.popStep();
        }

        for (ClassDecl nestedClass : cInsn.getNestedClasses()) {
            apply(nestedClass);
        }
    }

    private void cleanupTargetedInnerConstructorCall(MethodDecl ctor) {
        Invoke invoke = matchInvoke(ctor.getBody().getEntryPoint().getFirstChildOrNull(), InvokeKind.SPECIAL, "<init>");
        if (invoke == null) return;

        // Cleanup first argument to targeted super or this constructor calls.
        if (TypeSystem.isConstructedViaTargetInstance(invoke.getMethod().getDeclaringClass())) {
            Instruction first = invoke.getArguments().first();
            first.replaceWith(new Nop());
        }
    }

    private void removeRedundantSuperCall(MethodDecl ctor) {
        Invoke invoke = getSuperConstructorCall(ctor);
        if (invoke == null) return;

        // If all arguments are nop, its considered implicit.
        if (!invoke.getArguments().allMatch(e -> e instanceof Nop)) return;

        invoke.remove();
    }

    private void removeRedundantCanonicalRecordConstructor(ClassDecl cInsn) {
        MethodDecl canonicalCtor = cInsn.canonicalCtor;
        if (canonicalCtor == null) return;

        if (!isEmpty(canonicalCtor)) return;

        if (ColUtils.anyMatch(canonicalCtor.getMethod().getParameters(), Parameter::isMandated)) return;

        canonicalCtor.remove();
        cInsn.canonicalCtor = null;
    }

    private void removeRedundantConstructors(ClassDecl cInsn, List<MethodDecl> constructors) {
        if (constructors.size() != 1) return;
        MethodDecl ctor = constructors.get(0);

        if (cInsn.getClazz().getDeclType() == ClassType.DeclType.ANONYMOUS) {
            // Always yeet anonymous constructors.
            ctor.remove();
            return;
        }

        // If the ctor throws exceptions, We can't remove it.
        if (!ctor.getMethod().getExceptions().isEmpty()) return;

        AccessFlag access = AccessFlag.getAccess(ctor.getMethod().getAccessFlags());
        if (cInsn.getClazz().getDeclType() == ClassType.DeclType.LOCAL) {
            if (access != null) return;
        } else if (cInsn.getClazz().isEnum()) {
            // enum ctors are always private
        } else {
            if (access != AccessFlag.getAccess(cInsn.getClazz().getAccessFlags())) return;
        }

        if (!ctor.getMethod().getAnnotationSupplier().isEmpty()) return;
        if (!ctor.parameters.allMatch(ParameterVariable::isImplicit)) return;
        if (!isEmpty(ctor)) return;

        ctor.remove();
    }

    private static boolean isEmpty(MethodDecl function) {
        BlockContainer body = function.getBody();
        if (body.getEntryPoint().instructions.size() != 1) return false;
        if (BranchLeaveMatching.matchReturn(body.getEntryPoint().getFirstChild(), function) == null) return false;
        return true;
    }
}
