package net.covers1624.coffeegrinder.bytecode.transform.transformers.statement;

import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.SimpleInsnVisitor;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching;
import net.covers1624.coffeegrinder.bytecode.transform.StatementTransformContext;
import net.covers1624.coffeegrinder.bytecode.transform.StatementTransformer;
import net.covers1624.coffeegrinder.type.Method;
import net.covers1624.coffeegrinder.type.accessors.SyntheticAccessor;
import net.covers1624.coffeegrinder.type.accessors.SyntheticAccessor.CtorAccessor;
import net.covers1624.coffeegrinder.type.accessors.SyntheticAccessor.FieldAccessor;
import net.covers1624.coffeegrinder.type.accessors.SyntheticAccessor.FieldIncrementAccessor;
import net.covers1624.coffeegrinder.type.accessors.SyntheticAccessor.MethodAccessor;
import net.covers1624.coffeegrinder.util.None;
import net.covers1624.quack.collection.FastStream;

import static net.covers1624.coffeegrinder.type.accessors.SyntheticAccessor.AccessorType.FIELD_LOAD;
import static net.covers1624.coffeegrinder.type.accessors.SyntheticAccessor.AccessorType.FIELD_POST_INC;

/**
 * Created by covers1624 on 6/7/22.
 */
public class AccessorTransforms extends SimpleInsnVisitor<StatementTransformContext> implements StatementTransformer {

    @Override
    public void transform(Instruction statement, StatementTransformContext ctx) {
        statement.accept(this, ctx);
    }

    @Override
    public None visitInvoke(Invoke invoke, StatementTransformContext ctx) {
        process(invoke, ctx);
        return NONE;
    }

    @Override
    public None visitNew(New newInsn, StatementTransformContext ctx) {
        process(newInsn, ctx);
        return NONE;
    }

    private void process(AbstractInvoke invoke, StatementTransformContext ctx) {
        if (!invoke.getMethod().isSynthetic()) return;

        Method method = invoke.getMethod().getDeclaration();

        SyntheticAccessor accessor = method.getAccessor();
        if (accessor == null) return;

        ctx.pushStep("Replace accessor " + method.getName());
        replace(accessor, invoke, ctx);
        ctx.popStep();
    }

    private Instruction replace(SyntheticAccessor accessor, AbstractInvoke invoke, StatementTransformContext ctx) {
        switch (accessor.type) {
            case INVOKE:
                return replaceMethodAccessor((MethodAccessor) accessor, (Invoke) invoke);
            case FIELD_LOAD:
            case FIELD_STORE:
                return replaceFieldAccessor((FieldAccessor) accessor, (Invoke) invoke);
            case FIELD_POST_INC:
            case FIELD_PRE_INC:
                return replaceFieldIncrementAccessor((FieldIncrementAccessor) accessor, (Invoke) invoke);
            case CONSTRUCTOR:
                return replaceCtorAccessor((CtorAccessor) accessor, invoke, ctx);
            default:
                throw new IllegalStateException("Unhandled accessor type: " + accessor.type);
        }
    }

    private Instruction replaceMethodAccessor(MethodAccessor accessor, Invoke usage) {
        Instruction target = accessor.method.isStatic() ? new Nop() : usage.getArguments().first();
        return usage.replaceWith(new Invoke(
                accessor.method.isStatic() ? Invoke.InvokeKind.STATIC : Invoke.InvokeKind.VIRTUAL,
                accessor.method.asRaw(),
                target,
                usage.getArguments().skip(accessor.method.isStatic() ? 0 : 1)
        ));
    }

    private Instruction replaceFieldAccessor(FieldAccessor accessor, Invoke usage) {
        boolean isStatic = accessor.field.isStatic();
        FieldReference ref = new FieldReference(accessor.field.asRaw(), isStatic ? new Nop() : usage.getArguments().first());
        if (accessor.type == FIELD_LOAD) {
            return usage.replaceWith(new Load(ref));
        }
        Instruction value = isStatic ? usage.getArguments().first() : usage.getArguments().get(1);
        return usage.replaceWith(new Store(ref, value));
    }

    private Instruction replaceFieldIncrementAccessor(FieldIncrementAccessor accessor, Invoke usage) {
        boolean isStatic = accessor.field.isStatic();
        FieldReference ref = new FieldReference(accessor.field.asRaw(), isStatic ? new Nop() : usage.getArguments().first());
        if (accessor.type == FIELD_POST_INC) {
            return usage.replaceWith(new PostIncrement(ref, accessor.positive));
        }

        return usage.replaceWith(new CompoundAssignment(accessor.positive ? BinaryOp.ADD : BinaryOp.SUB, ref, new LdcNumber(1)));
    }

    private Instruction replaceCtorAccessor(CtorAccessor accessor, AbstractInvoke usage, StatementTransformContext ctx) {
        // Nuke the un-inlined null.
        Load load = LoadStoreMatching.matchLoad(usage.getArguments().last());
        assert load != null;
        Store store = LoadStoreMatching.matchStoreLocal(usage.getPrevSibling(), load.getVariable());
        assert store != null;
        assert store.getValue() instanceof LdcNull;
        ctx.moveNext();
        store.remove();

        FastStream<Instruction> headArgs = usage.getArguments().limit(usage.getArguments().size() - 1);
        if (usage instanceof Invoke invoke) {
            // Super calls.
            assert invoke.getKind() == Invoke.InvokeKind.SPECIAL;

            return usage.replaceWith(new Invoke(Invoke.InvokeKind.SPECIAL, accessor.method.asRaw(), invoke.getTarget(), headArgs));
        }
        return usage.replaceWith(new New(((New) usage).getResultType(), accessor.method.asRaw(), headArgs));

    }
}
