package net.covers1624.coffeegrinder.source;

import net.covers1624.coffeegrinder.bytecode.AccessFlag;
import net.covers1624.coffeegrinder.bytecode.insns.ClassDecl;
import net.covers1624.coffeegrinder.type.*;
import net.covers1624.coffeegrinder.type.AnnotationSupplier.TypeAnnotationLocation;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;

import java.util.*;

import static net.covers1624.coffeegrinder.source.LineBuffer.of;
import static net.covers1624.coffeegrinder.type.TypeAnnotationData.EMPTY;

/**
 * Created by covers1624 on 8/8/21.
 */
// TODO Import collector needs to know about class declaration scopes.
public class ImportCollector {

    private final @Nullable TypeResolver typeResolver;
    private @Nullable NumericConstantPrinter constantPrinter;

    private final LinkedList<ClassType> scopeStack = new LinkedList<>();
    private final LinkedList<HashSet<String>> variableNameStack = new LinkedList<>();
    private final List<String> imports = new ArrayList<>();
    private final Set<String> aliases = new HashSet<>();
    private final Map<String, String> aliasLookup = new HashMap<>();

    @Nullable
    private ClassType topLevelClass;

    public ImportCollector(@Nullable TypeResolver typeResolver) {
        this.typeResolver = typeResolver;
    }

    public void setConstantPrinter(NumericConstantPrinter printer) {
        this.constantPrinter = printer;
    }

    public @Nullable NumericConstantPrinter getConstantPrinter() {
        return constantPrinter;
    }

    public List<String> getImports(@Nullable ClassDecl context) {
        return FastStream.of(imports)
                .sorted() // TODO, we need some sort of package grouping for imports.
                .filter(e -> context == null || !e.equals(context.getClazz().getFullName()))
                .toLinkedList();
    }

    public void setTopLevelClass(ClassType topLevelClass) {
        this.topLevelClass = topLevelClass;
        // Trigger collection of top-level class, reserving the short class name.
        collect(topLevelClass);
    }

    public void pushScope(ClassType type) {
        scopeStack.push(type);
    }

    public void popScope() {
        scopeStack.pop();
    }

    public void pushVariableScope() {
        variableNameStack.push(new HashSet<>());
    }

    public boolean isVariableInScope(String variableName) {
        for (HashSet<String> names : variableNameStack) {
            if (names.contains(variableName)) {
                return true;
            }
        }
        return false;
    }

    public void pushVariableName(String name) {
        Set<String> namesForStack = variableNameStack.peek();
        assert namesForStack != null;
        namesForStack.add(name);
    }

    public void popVariableScope() {
        variableNameStack.pop();
    }

    public String collectSimpleTypeParam(TypeParameter parameter) {
        if (TypeSystem.isObject(parameter.getUpperBound())) {
            return parameter.getName();
        }
        return parameter.getName() + " extends " + collect(parameter.getUpperBound(), EMPTY);
    }

    public String collectTypeParam(TypeParameter parameter, AnnotationSupplier supplier) {
        String ann = typeAnnotations(supplier.getTypeAnnotations(TypeAnnotationLocation.TYPE_PARAMETER, null, parameter.getIndex()));
        TypeAnnotationData boundAnnotation = supplier.getTypeAnnotations(TypeAnnotationLocation.TYPE_PARAMETER_BOUND, parameter.getUpperBound(), parameter.getIndex());
        // T extends Object is redundant only if we have no annotation to apply.
        if (TypeSystem.isObject(parameter.getUpperBound()) && boundAnnotation.isEmpty()) {
            return ann + parameter.getName();
        }
        return ann + parameter.getName() + " extends " + collect(parameter.getUpperBound(), boundAnnotation);
    }

    public final LineBuffer annotations(Iterable<AnnotationData> annotations) {
        LineBuffer buffer = of();
        for (AnnotationData annotation : annotations) {
            buffer = buffer.join(annotation(annotation));
        }
        return buffer;
    }

    private String typeAnnotations(TypeAnnotationData data) {
        if (data.isEmpty()) return "";

        LineBuffer buffer = annotations(data.annotations);
        String str = buffer.joinOn(" ").toString();
        if (str.isEmpty()) return "";
        return str + " ";
    }

    private LineBuffer annotation(AnnotationData data) {
        LineBuffer buffer = of("@").append(collect(data.type));
        if (data.isEmpty()) return buffer;

        buffer = buffer.append(" (");
        if (data.isSingleElement()) {
            buffer = buffer.append(annotationValue(data.values.get("value")));
        } else {
            boolean first = true;
            for (Map.Entry<String, Object> entry : data.values.entrySet()) {
                if (!first) {
                    buffer = buffer.append(", ");
                }
                first = false;
                buffer = buffer.append(entry.getKey()).append(" = ").append(annotationValue(entry.getValue()));
            }
        }
        buffer = buffer.append(")");

        return buffer;
    }

    public LineBuffer annotationValue(Object obj) {
        if (obj instanceof ClassType || obj instanceof ArrayType) return of(collect((AType) obj)).append(".class");
        if (obj instanceof String) return of("\"").append(EscapeUtils.escapeChars(obj.toString())).append("\"");
        if (obj instanceof Character) return of("'").append(EscapeUtils.escapeChar((Character) obj)).append("'");
        if (obj instanceof Number) {
            if (constantPrinter != null) return constantPrinter.printNumber((Number) obj);
            return LineBuffer.of(obj.toString());
        }
        if (obj instanceof AnnotationData) return annotation((AnnotationData) obj);

        if (obj instanceof Field) {
            Field field = (Field) obj;
            return of(collect(field.getDeclaringClass())).append(".").append(field.getName());
        }

        if (obj instanceof List) {
            List<?> list = (List<?>) obj;
            if (list.isEmpty()) return of("{}");
            // Arrays with single elements can have the array decl omitted.
            if (list.size() == 1) return of(annotationValue(list.get(0)));

            LineBuffer buffer = of("{");
            boolean first = true;
            for (Object o : list) {
                if (!first) {
                    buffer = buffer.append(", ");
                }
                first = false;
                buffer = buffer.append(annotationValue(o));
            }
            return buffer.append("}");
        }
        return of(obj.toString());
    }

    public String collect(AType type) {
        return collect(type, EMPTY);
    }

    public String collect(AType type, TypeAnnotationData annotations) {
        String annRet = typeAnnotations(annotations);
        if (type instanceof PrimitiveType || type instanceof IntegerConstantType || type instanceof IntegerConstantUnion || type instanceof NullConstantType) {
            return annRet + type.getFullName();
        }

        if (type instanceof ArrayType) {
            return collect(((ArrayType) type).getElementType(), annotations.step(TypeAnnotationData.Target.ARRAY_ELEMENT)) + padStart(annRet) + "[]";
        }

        if (type instanceof CapturedTypeVar) {
            return "capture of " + collect(((CapturedTypeVar) type).wildcard, EMPTY);
        }

        if (type instanceof TypeVariable) {
            return annRet + type.getName();
        }

        if (type instanceof ReferenceUnionType) {
            return annRet + FastStream.of(((ReferenceUnionType) type).getTypes()).map(this::collect).join(" | ");
        }

        if (type instanceof IntersectionType) {
            IntersectionType intersection = (IntersectionType) type;

            return annRet + intersection.getDirectSuperTypes()
                    .map(this::collect)
                    .join(" & ");
        }

        if (type instanceof WildcardType) {
            WildcardType arg = (WildcardType) type;
            TypeAnnotationData wildcardData = annotations.step(TypeAnnotationData.Target.WILDCARD_BOUND);
            if (arg.isInfinite()) {
                return "...";
            }
            if (arg.isSuper()) {
                return annRet + "? super " + collect(arg.getLowerBound(), wildcardData);
            }
            if (TypeSystem.isObject(arg.getUpperBound()) && wildcardData.isEmpty()) {
                return annRet + "?";
            }
            return annRet + "? extends " + collect(arg.getUpperBound(), wildcardData);
        }

        assert type instanceof ClassType : "Unhandled type: " + type.getClass().getName();

        return collect((ClassType) type, annotations);
    }

    public String collect(ClassType type) {
        return collect(type, EMPTY);
    }

    public String collect(ClassType type, TypeAnnotationData annotations) {
        return collect(type, annotations, false, false);
    }

    public String collect(ClassType type, TypeAnnotationData annotations, boolean targeted, boolean inferredTypeArgs) {
        String annRet = typeAnnotations(annotations);
        if (type instanceof ParameterizedClass) {
            ParameterizedClass pClass = (ParameterizedClass) type;
            String ret;
            if (targeted) {
                ret = pClass.getName();
            } else if (pClass.getDeclType() != ClassType.DeclType.LOCAL && pClass.getOuter() != null) {
                ret = collect(pClass.getOuter(), annotations.step(TypeAnnotationData.Target.OUTER_TYPE)) + "." + annRet + pClass.getName();
                annRet = "";
            } else {
                annRet = "";
                ret = collect(pClass.getDeclaration(), annotations);
            }
            List<ReferenceType> typeArguments = pClass.getTypeArguments();
            if (!typeArguments.isEmpty()) {
                if (inferredTypeArgs) {
                    ret += "<>";
                } else {
                    StringBuilder b = new StringBuilder("<");
                    for (int i = 0; i < typeArguments.size(); i++) {
                        if (i != 0) {
                            b.append(", ");
                        }
                        b.append(collect(typeArguments.get(i), annotations.step(TypeAnnotationData.Target.TYPE_ARGUMENT, i)));
                    }
                    b.append(">");
                    ret += b;
                }
            }
            return annRet + ret;
        }

        if (targeted || type.getDeclType().isLocalOrAnonymous()) {
            return type.getName();
        }

        Optional<ClassType> outerClass = type.getEnclosingClass();
        if (outerClass.isPresent()) {
            TypeAnnotationData outerAnnotations = annotations.step(TypeAnnotationData.Target.OUTER_TYPE);
            if (outerAnnotations.isEmpty() && !isTypeHidden(type) && scopeStack.contains(outerClass.get())) {
                return annRet + type.getName();
            }
            return collect(outerClass.get(), outerAnnotations) + "." + annRet + type.getName();
        }

        String fullName = type.getFullName();

        // Does a field/local hide this type name.
        boolean hidden = isTypeHidden(type);
        if (hidden) return annRet + fullName;

        String alias = aliasLookup.get(fullName);
        // Have we already assigned this type a name?
        if (alias != null) return annRet + alias;

        alias = type.getName();
        // Has this alias been used
        if (!aliases.add(alias)) {
            // Alias already used, must use full type name.
            alias = fullName;
        } else if (!isImplicitlyImported(type)) { // Can we import you?
            imports.add(fullName);
        }
        aliasLookup.put(fullName, alias);
        return annRet + alias;
    }

    public boolean isTypeHidden(ClassType type) {
        if (isVariableInScope(type.getName())) return true;
        if (findFieldInScope(type.getName()) != null) return true;
        ClassType foundInner = findInnerInScope(type.getName());
        if (foundInner != null && !foundInner.equals(type.getDeclaration())) return true;
        return false;
    }

    public boolean isFieldHidden(@Nullable ClassType target, Field field) {
        // A LocalVariable has hidden it, must be qualified.
        if (isVariableInScope(field.getName())) return true;

        var found = findFieldInScope(field.getName());
        if (found != null) {
            return (target != null && found.scope != target) || found.field != field.getDeclaration();
        }

        // We don't know. Assume the access must be fully qualified.
        return true;
    }

    public boolean doesStaticFieldRequireQualifier(@Nullable ClassType targetClassType, Field field) {
        return isFieldHidden(null, field) || targetClassType != null && !scopeStack.contains(targetClassType);
    }

    public boolean isMethodHidden(ClassType targetClassType, String methodName) {
        if (scopeStack.isEmpty()) return true;

        assert scopeStack.contains(targetClassType);
        for (ClassType scope : scopeStack) {
            if (scope == targetClassType) return false;

            if (findMethodInHierarchy(scope, methodName) != null) {
                return true;
            }
        }

        throw new IllegalStateException();
    }

    public boolean doesStaticMethodRequireQualifier(ClassType targetClassType, String methodName) {
        return !scopeStack.contains(targetClassType) || isMethodHidden(targetClassType, methodName);
    }

    @Nullable
    private FieldInScope findFieldInScope(String name) {
        for (ClassType scope : scopeStack) {
            Field found = findFieldInHierarchy(scope, name);
            if (found != null) {
                return new FieldInScope(scope, found);
            }
        }

        return null;
    }

    @Nullable
    private ClassType findInnerInScope(String name) {
        for (ClassType scope : scopeStack) {
            ClassType innerClass = findInnerInHierarchy(scope, name);
            if (innerClass != null) {
                return innerClass;
            }
        }
        return null;
    }

    private boolean isImplicitlyImported(ClassType type) {
        String pkg = type.getPackage();
        String packageScope = topLevelClass != null ? topLevelClass.getPackage() : null;

        if (packageScope != null) {
            // Current scope contains type to be imported. Is implicit.
            if (packageScope.equals(pkg)) return true;

            // Scope contains a type with the same alias as the type to be imported. Not implicit.
            if (typeResolver != null && typeResolver.classExists(packageScope + "." + type.getName())) {
                return false;
            }
        }

        // java.lang is implicit.
        return pkg.equals("java.lang");
    }

    @Nullable
    private ClassType findInnerInHierarchy(ClassType type, String name) {
        for (ClassType nestedClass : type.getNestedClasses()) {
            if (nestedClass.getName().equals(name)) {
                return nestedClass;
            }
        }

        return type.getDirectSuperTypes()
                .map(e -> findInnerInHierarchy(e.getDeclaration(), name))
                .filter(Objects::nonNull)
                .firstOrDefault();
    }

    @Nullable
    private static Field findFieldInHierarchy(ClassType type, String name) {
        for (Field field : type.getDeclaration().getFields()) {
            if (field.getName().equals(name)) {
                return field;
            }
        }

        return type.getDirectSuperTypes()
                .map(e -> findFieldInHierarchy(e, name))
                .filter(Objects::nonNull)
                .filterNot(e -> e.getAccessFlags().get(AccessFlag.PRIVATE))
                .firstOrDefault();
    }

    @Nullable
    private static Method findMethodInHierarchy(ClassType type, String name) {
        for (Method method : type.getDeclaration().getMethods()) {
            if (method.getName().equals(name)) {
                return method;
            }
        }

        return type.getDirectSuperTypes()
                .map(e -> findMethodInHierarchy(e, name))
                .filter(Objects::nonNull)
                .firstOrDefault();
    }

    private static String padStart(String str) {
        return str.isEmpty() ? str : " " + str;
    }

    private record FieldInScope(ClassType scope, Field field) { }
}
